//
//  KonyPKPass.h
//  KLPassBook
//
//  Created by Amba on 06/01/14.
//  Copyright (c) 2014 Konylabs. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <UIKit/UIKit.h>

@class PKPass;
/*!
 The KonyPKPass class represents a single pass.
 @note KonyPKPass is a wrapper class around PKPass
 */
@interface KonyPKPass : NSObject

/*!
 @abstract Initializes and returns an instance of the class.
 
 @param pass A PKPass instance.
 
 @return The newly initialized instance.
 */
- (id)initWithPKPass:(PKPass*)pass;

/*! @name Identifying a Pass */
/*!
 @abstract Creates and returns an instance of the class.
 
 @throws NSException with KonyPKError name if unable to create pass
 
 @param content A definition of a pass, as described in Passbook Package Format Reference.
 
 @return The newly created instance with passcontent.
 */
+ (KonyPKPass *)konyPKPassWithContent:(NSData *)content;

/*!
 
 @abstract Creates and returns an instance of the class.
 
 @throws NSException with KonyPKError name if unable to create pass
 
 @param path A path to file which has definition of a pass, as described in Passbook Package Format Reference.
 
 @return The newly created instance with pass from file.
 
 @discussion The path to a pkpass file is relative in the application sandbox.
 */
+ (KonyPKPass *)konyPKPassFromFilePath:(NSString *)path;

/*!
 
 @abstract The URL that opens the pass in the Passbook app.
 
 @return absolute string of passURL
 
 @discussion Open the URL that is returned by passing it as the argument of the openURL: method of UIApplication.
 */
-(NSString *)passURL;

/*! @name Identifying a Pass */
/*!
 
 @abstract A value that uniquely identifies the pass.
 
 @return The serial number of Pass
 */
- (NSString *)serialNumber;

/*!
 @abstract The pass’s type identifier.
 
 @return The pass’s type identifier.
 
 @discussion For information about getting a pass type identifier, see Passbook Programming Guide.
 */
- (NSString *)passTypeIdentifier;

/*!
 @abstract The URL for your web service.
 
 @return The URL for your web service.
 
 @discussion The web service is used to update passes. For information on the the web service protocol, see Passbook Web Service Reference.
 */
- (NSString *)webServiceURL;

/*!
 @abstract The token used to authenticate with the web service.
 
 @return authenticationToken of PKPass
 
 @discussion An app that uses its own protocol for communicating with your server about passes can use this. The app can include the authentication token in its requests to show that they are coming from a device that has a copy of the pass, and not from an attacker.
 */
- (NSString *)authenticationToken;

/*! @name Displaying a Pass */
/*!
 @abstract The pass’s icon.
 
 @return icon of PKPass.
 
 @discussion You can use this to display a pass’s icon as part of a UI element that represents the pass, such as a cell in a table view.
 
 @todo return RawBytes - with NSData
 */
- (UIImage *)icon;

/*!
 @abstract You can use this property to provide accessibility information for a UI element that represents a pass, such as a cell in a table view.
 
 @return localizedName of PKPass.
 
 @discussion You can use this property to provide accessibility information for a UI element that represents a pass, such as a cell in a table view.
 */
- (NSString *)localizedName;

/*!
 @abstract The localized description of the pass’s kind.
 
 @return localizedDescription of PKPass.
 
 @discussion You can use this property to provide accessibility information for a UI element that represents a pass, such as a cell in a table view.
 */
- (NSString *)localizedDescription;

/*!
 @abstract The name of the organization that created the pass.
 
 @return organizationName of PKPass.
 
 @discussion You can use this property to display information about a pass as part of a UI element that represents the pass, such as a cell in a table view.
 */
- (NSString *)organizationName;

/*!
 @abstract The date when the pass is most likely to be needed or useful.
 
 @return relevantDate of PKPass.
 
 @discussion You can use this property for sorting UI elements that represent passes, such as cells in a table view.
 */
- (NSString *)relevantDate;

/*!
 @abstract Developer-specific custom data.
 @return userInfo of PKPass.
 */
- (NSDictionary *)userInfo;

/*!
 @abstract Returns the localized value for specified field of the pass.
 
 @note Localization here is device localization
 
 @param key The field's key, as specified in the pass.
 
 @return The localized value for the pass’s field.
 
 @discussion If your app works with passes from arbitrary sources, such as an email client, it can’t use this method because the passes’ keys are not known ahead of time. Use the other properties of this class, such as icon and organizationName, instead.
 */
- (id) localizedValueForKeyForDeviceLocale:(NSString *)key;

@end


/*! @name Error Messages */
/*! KonyPKUnknownError Unknown error.*/
extern NSString *const KonyPKUnknownError;

/*! KonyPKInvalidDataError Invalid pass data.*/
extern NSString *const KonyPKInvalidDataError;

/*! KonyPKUnsupportedVersionError Unsupported pass version.*/
extern NSString *const KonyPKUnsupportedVersionError;

/*! KonyPKInvalidSignature Pass signature is invalid.*/
extern NSString *const KonyPKInvalidSignature;
