//
//  KonyPKPassLibrary.h
//  KLPassBook
//
//  Created by Amba on 06/01/14.
//  Copyright (c) 2014 Konylabs. All rights reserved.
//

#import <Foundation/Foundation.h>

@class KonyPKPass;
@class CallBack;

/*!
 The KonyPKPassLibrary class provides an interface to the user’s library of passes.
 
 @note KonyPKPassLibrary is a wrapper class around PKPassLibrary.
 */
@interface KonyPKPassLibrary : NSObject {
}

/*!
 @abstract Returns whether the pass library is available.
 
 @return YES if the pass library and PKPassLibrary class is available; otherwise, NO.
 
 @discussion This method checks PKPassLibrary class existstance and the pass library availability.
 */
+ (BOOL)isPassLibraryAvailable;

/*!
 @abstract Returns the passes in the user’s pass library.
 
 @return The passes in the user’s pass library.
 
 @discussion Your app only has access to certain passes, based on its entitlements. Passes that your app doesn’t have access to are not returned.
 The ordering of the passes is not fixed; calling this method multiple times may return the same passes but in a different order.
 */
- (NSArray *)passes;

/*!
 @abstract Returns the pass with the given pass type identifier and serial number.
 
 @param identifier The pass’s pass type identifier.
 @param serialNumber The pass’s serial number.
 
 @return The pass with the given pass type identifier and serial number, or nil if there is no such pass or if the app doesn’t have the appropriate entitlement.
 */
- (KonyPKPass *)passWithPassTypeIdentifier:(NSString *)identifier andSerialNumber:(NSString *)serialNumber;

/*!
 @abstract Presents UI to add pass.
 
 @param pass The pass being added.
 @param callback The completion callback called after the user. This callback takes the following parameters:
 
 @discussion status Indicates whether the pass got added. If the status is KonyPKPassLibraryShouldReviewPasses, present an instance of KonyPKAddPassesViewController with the same pass, to let the user review and add them.
 */
- (void)addPass:(KonyPKPass *)pass withCompletionCallback:(CallBack *)callback;

/*!
 @abstract Presents UI to add multiple passes.
 
 @param passes The passes being added.
 @param callback The completion callback called after the user. This callback takes the following parameters:
 status Indicates whether the passes were added. If the status is KonyPKPassLibraryShouldReviewPasses, present an instance of KonyPKAddPassesViewController with the same pass, to let the user review and add them.
 */
- (void)addPasses:(NSArray *)passes withCompletionCallback:(CallBack *)callback;

/*!
 @abstract Removes the pass from the user’s pass library.
 
 @param pass The pass being removed.
 
 @discussion This method does nothing if your app doesn’t have the appropriate entitlement.
 */
- (void)removePass:(KonyPKPass *)pass;

/*!
 @abstract Replaces a pass in the user’s pass library with the given pass.
 
 @param pass The new pass.
 
 @return YES if the pass was replaced successfully; otherwise NO.
 
 @discussion The new pass replaces the existing pass with the same pass type identifier and serial number. If there is no such pass in the user’s pass library, the replacement fails.
 */
- (BOOL)replacePassWithPass:(KonyPKPass *)pass;

/*!
 @abstract Returns whether the user’s pass library contains a pass.
 
 @param pass The pass being queried.
 
 @return YES if the user’s pass library contains the pass; otherwise, NO.
 
 @discussion This method lets you determine that the pass library contains a pass even though your app wouldn’t be able to read or modify the pass. For example, an email client would not have entitlements to read or write any passes from the library.
 Your app can use this method to provide UI indicating whether a pass is already in the library.
 */
- (BOOL)containsPass:(KonyPKPass *)pass;

@end

/*!
 KonyPKPassLibraryDidAddPasses Status used when passes were sucessfully added.
 */
extern NSString *const KonyPKPassLibraryDidAddPasses;

/*!
 KonyPKPassLibraryShouldReviewPasses Status used when the app should display UI to let the user review the passes.
 */
extern NSString *const KonyPKPassLibraryShouldReviewPasses;

/*!
 KonyPKPassLibraryDidCancelAddPasses Status used when the user cancelled and the passes were not added.
 */
extern NSString *const KonyPKPassLibraryDidCancelAddPasses;
