#!/bin/bash
# Define the network IP range here. This value may need to be updated depending on the target environment to avoid IP conflicts.
# Configurable network range 
# - Host in 192.x.x.x ? use 10.x.x.x CIDR range
# - Host in 10.x.x.x or 172.x.x.x ? use 192.x.x.x CIDR range
# Set this based on your network
POD_NETWORK_CIDR="172.28.0.0/20"   # Change if needed

#PRE-Reqs: Disable Swap | Bridge Traffic (Run it on MASTER & WORKER Nodes):
# Disable Swap (Required for Kubernetes) - Both Master and Worker-node
sudo swapoff -a

# Disable swap NOW and prevent it at next boot
sudo sed -i '/\s\+swap\s\+/ s/^/#/' /etc/fstab 
(crontab -l 2>/dev/null; echo "@reboot /sbin/swapoff -a") | crontab -

# Set SELinux to Permissive Mode (Required for Kubernetes)
sudo sed -i 's/^SELINUX=enforcing$/SELINUX=permissive/' /etc/selinux/config
sudo setenforce 0

# Install iproute-tc
sudo dnf install -y iproute-tc

# Configure Kernel Modules for Kubernetes
cat <<EOF | sudo tee /etc/modules-load.d/k8s.conf
overlay
br_netfilter
EOF

# Load necessary Kernel modules
sudo modprobe overlay
sudo modprobe br_netfilter

# Configure Sysctl settings for Kubernetes
cat <<EOF | sudo tee /etc/sysctl.d/k8s.conf
net.bridge.bridge-nf-call-iptables = 1
net.ipv4.ip_forward = 1
net.bridge.bridge-nf-call-ip6tables = 1
net.ipv4.conf.all.rp_filter = 0
EOF

# Apply Sysctl settings
sudo sysctl --system

# Install CRI-O (Container Runtime for Kubernetes)
sudo yum -y update
sudo subscription-manager repos --enable codeready-builder-for-rhel-9-$(arch)-rpms
sudo dnf install -y https://dl.fedoraproject.org/pub/epel/epel-release-latest-9.noarch.rpm
KUBERNETES_VERSION=v1.31
PROJECT_PATH=prerelease:/main

# Add Kubernetes and CRI-O repos
cat <<EOF | sudo tee /etc/yum.repos.d/kubernetes.repo
[kubernetes]
name=Kubernetes
baseurl=https://pkgs.k8s.io/core:/stable:/$KUBERNETES_VERSION/rpm/
enabled=1
gpgcheck=1
gpgkey=https://pkgs.k8s.io/core:/stable:/$KUBERNETES_VERSION/rpm/repodata/repomd.xml.key
EOF

cat <<EOF | sudo tee /etc/yum.repos.d/cri-o.repo
[cri-o]
name=CRI-O
baseurl=https://pkgs.k8s.io/addons:/cri-o:/$PROJECT_PATH/rpm/
enabled=1
gpgcheck=1
gpgkey=https://pkgs.k8s.io/addons:/cri-o:/$PROJECT_PATH/rpm/repodata/repomd.xml.key
EOF

# Install dependencies and CRI-O container runtime
sudo dnf install -y container-selinux
sudo dnf install -y cri-o
sudo systemctl start crio.service
sudo systemctl enable --now crio
sudo systemctl status crio  | grep Active

# Install Helm
curl -fsSL -o get_helm.sh https://raw.githubusercontent.com/helm/helm/master/scripts/get-helm-3
chmod +x get_helm.sh
./get_helm.sh

# Add /usr/local/bin to all users' PATH
sudo grep -q '/usr/local/bin' /etc/sudoers || \
echo 'Defaults secure_path="/usr/local/sbin:/usr/local/bin:/sbin:/bin:/usr/sbin:/usr/bin"' | sudo tee -a /etc/sudoers.d/01-custom-path

# Update PATH for the current shell
export PATH="/usr/local/bin:$PATH"

# Verify Helm installation
helm version

# Install Kubeadm
sudo setenforce 0
cat <<EOF | sudo tee /etc/yum.repos.d/kubernetes.repo
[kubernetes]
name=Kubernetes
baseurl=https://pkgs.k8s.io/core:/stable:/$KUBERNETES_VERSION/rpm/
enabled=1
gpgcheck=1
gpgkey=https://pkgs.k8s.io/core:/stable:/$KUBERNETES_VERSION/rpm/repodata/repomd.xml.key
exclude=kubelet kubeadm kubectl cri-tools kubernetes-cni
EOF

sudo yum install -y kubelet kubeadm kubectl --disableexcludes=kubernetes
sudo systemctl enable --now kubelet

#Initializing CONTROL-PLANE (Run it on MASTER Node only)
# Initialize Kubernetes cluster with kubeadm
sudo kubeadm init --pod-network-cidr="$POD_NETWORK_CIDR"

sudo mkdir -p /etc/kubeconfigs
sudo cp /etc/kubernetes/admin.conf /etc/kubeconfigs/admin.conf
export KUBECONFIG=/etc/kubeconfigs/admin.conf
sudo chown $(id -u):$(id -g) /etc/kubeconfigs/admin.conf

# Allow scheduling on the master node
kubectl taint nodes --all node-role.kubernetes.io/control-plane- 

#Installing POD-NETWORK add-on (Run it on MASTER Node only)
# Install Calico network plugin for Kubernetes
kubectl apply -f https://raw.githubusercontent.com/projectcalico/calico/v3.25.0/manifests/calico.yaml

# Check the status of Kubernetes pods
kubectl get pods -n kube-system

# Install NFS server
sudo yum -y install nfs-utils
sudo mkdir -p /srv/nfs/ccm
sudo chown -R nobody /srv/nfs/ccm

# Configure NFS server
echo "/srv/nfs/ccm *(rw,sync,no_subtree_check,no_root_squash,no_all_squash,insecure)" | sudo tee -a /etc/exports

# Enable and start NFS server
sudo systemctl enable nfs-server
sudo systemctl restart nfs-server
sudo systemctl status nfs-server  | grep Active

# Export NFS shares
sudo exportfs -rav 

# Install NFS Subdir External Provisioner for Kubernetes
helm repo add nfs-subdir-external-provisioner https://kubernetes-sigs.github.io/nfs-subdir-external-provisioner/
helm repo update
helm install nfs-subdir-external-provisioner nfs-subdir-external-provisioner/nfs-subdir-external-provisioner --set nfs.server=$(hostname -I | awk '{print $1}') --set nfs.path=/srv/nfs/ccm --set storageClass.defaultClass=true

# Install Java 17
sudo yum install -y java-17-openjdk
java -version 

# Install Maven
sudo yum install -y maven 
source ~/.bashrc
mvn -version 

# Install Node.js and NPM
sudo yum install -y nodejs
node -v
npm -v

# Install Git
sudo yum install -y git
git --version 

# Install Docker
sudo yum install -y yum-utils
sudo yum-config-manager --add-repo https://download.docker.com/linux/rhel/docker-ce.repo
sudo yum install -y docker-ce docker-ce-cli containerd.io docker-buildx-plugin docker-compose-plugin 

# Start Docker
sudo systemctl start docker 

if [[ -n "$USERNAME" && -n "$PASSWORD" ]]; then
    echo "USERNAME or PASSWORD are set. Logging in to Docker non-interactively..."
    sudo su -c "echo \"$PASSWORD\" | docker login --username $USERNAME --password-stdin"
else
    echo "USERNAME or PASSWORD not set. Docker login"
    sudo su -c "docker login"
fi

#Disable firewall 
sudo systemctl stop firewalld
sudo systemctl disable firewalld

# Install emissary
helm repo add datawire https://app.getambassador.io
helm repo update

kubectl create namespace emissary && \
kubectl apply -f https://app.getambassador.io/yaml/emissary/3.9.1/emissary-crds.yaml

kubectl wait --timeout=200s --for=condition=available deployment emissary-apiext -n emissary-system

sudo mv /etc/cni/net.d/10-crio-bridge.conflist.disabled /etc/cni/net.d/10-crio-bridge.conflist

helm install emissary-ingress --namespace emissary datawire/emissary-ingress --skip-crds && \
kubectl -n emissary wait --for condition=available --timeout=120s deploy -lapp.kubernetes.io/instance=emissary-ingress 

kubectl get pods -n emissary -l app.kubernetes.io/instance=emissary-ingress 

kubectl get nodes 

kubectl patch service emissary-ingress -n emissary -p "{\"spec\": {\"externalIPs\": [\"$(hostname -I | awk '{print $1}')\"]}}" 

# Setup kubectl access for current user 
mkdir -p $HOME/.kube
sudo cp -i /etc/kubernetes/admin.conf $HOME/.kube/config
sudo chown $(id -u):$(id -g) $HOME/.kube/config
echo 'export KUBECONFIG=$HOME/.kube/config' >> ~/.bashrc
source ~/.bashrc 

echo "Restarting pods that are currently not in a Ready state..."
kubectl get pods --all-namespaces --field-selector=status.phase!=Running -o json | \
	jq -r '.items[] | "kubectl delete pod \(.metadata.name) -n \(.metadata.namespace)"' | \
	bash

echo "Waiting for pods to initialize and reach Running state..."
elapsed=0
while [ $elapsed -lt 120 ]; do
  kubectl get pods --all-namespaces | grep -vE 'Running|Completed|STATUS' | grep -q . || { echo "All pods Ready"; break; }
  sleep 15
  elapsed=$((elapsed + 15))
done

echo "Fetching current status of Kubernetes cluster:"
kubectl get pods -A

echo "Kubernetes Setup Completed Successfully."

echo "Note:"
echo "If you have not used a licensed Docker username and password, you may experience image pull back-off errors or pods getting stuck in the Init state."
echo "As a result, the Kubernetes cluster may take up to approximately 6-10 hours for all pods to become fully operational."