#Required
#External fully qualified domain name of the cluster
#
#See Installation of DevOps Loop documentation for more details:
#https://help.hcl-software.com/devops/loop/2.0.0/docs/topics/c_install.html
DOMAIN=$(hostname -I | awk '{ print $1; }').nip.io

#Required
#Set value to 'true' to accept the license.
ACCEPT_LICENSE=true

#Required
#Hostname of HCL MHS license server eg. https://hcl.com:443
LICENSE_SERVER=
#License Pool ID (obtained from MHS)
LICENSE_ID=

#Required
#SMTP Server configuration
EMAIL_SERVER_HOST=
EMAIL_SERVER_PORT=
EMAIL_FROM_ADDRESS=

#Optional
#Additional SMTP Server configuration
EMAIL_SERVER_USERNAME=""
EMAIL_SERVER_PASSWORD=""
EMAIL_SERVER_STARTTLS=false
EMAIL_SERVER_SMTPS=false

#Required
#Secure seed required to generate passwords
#A random string
#Unrecoverable so keep it safe
PASSWORD_SEED=secret

#Specify the TLS secret name in your namespace as necessary.
#
#See Installation of DevOps Loop documentation for more details:
#https://help.hcl-software.com/devops/loop/2.0.0/docs/topics/c_install.html
TLS_CERT_SECRET_NAME=devops-loop-tls-secret

#Set SELF_SIGNED=true to generate and use a self-signed certificate for the
#installation of DevOps Loop
SELF_SIGNED=true

#Alternative RWX storage class
#See Storage
#For example: ibmc-file-gold-gid
RWX_STORAGE_CLASS=nfs-client

#Alternative RWO storage class
#See Storage
#For example: ibmc-block-gold
RWO_STORAGE_CLASS=nfs-client

#Required
NAMESPACE=devops-loop
HELM_NAME=devops-loop
LOOP_CHART_VERSION=2.0.000

#Optional Additional Helm options
ADDITIONAL_HELM_OPTIONS=""

run_install() {

  if ! kubectl  get namespace ${NAMESPACE} > /dev/null 2>&1; then
    kubectl  create namespace ${NAMESPACE} || { echo "Failed to create namespace"; return 1; }
  fi


  if ! kubectl get secret mongodb-url-secret --namespace ${NAMESPACE} > /dev/null 2>&1; then

    MONGO_CHART_VERSION="${MONGO_CHART_VERSION:-14.13.0}"
    MONGO_IMAGE_TAG="${MONGO_IMAGE_TAG:-6.0}"
    MONGO_IMAGE_REPO="${MONGO_IMAGE_REPO:-bitnamilegacy/mongodb}"
    MONGO_HELM_RELEASE_NAME="${MONGO_HELM_RELEASE_NAME:-devops-loop-mongo}"
    MONGO_NAMESPACE="${MONGO_NAMESPACE:-${NAMESPACE}}"
    MONGO_PVC_SIZE=${MONGO_PVC_SIZE:-20Gi}

    helm repo add bitnami https://charts.bitnami.com/bitnami --force-update 1> /dev/null

    MONGO_INSTALL_OPTIONS="\
      --set image.repository=${MONGO_IMAGE_REPO} \
      --set image.tag=${MONGO_IMAGE_TAG} \
      --set persistence.size=${MONGO_PVC_SIZE}
    "

    if [ -n "${RWO_STORAGE_CLASS}" ]; then
      MONGO_INSTALL_OPTIONS="${MONGO_INSTALL_OPTIONS} --set persistence.storageClass=${RWO_STORAGE_CLASS}"
    fi

    if [ -n "${MONGO_ADDITIONAL_INSTALL_OPTIONS}" ]; then
      MONGO_INSTALL_OPTIONS="${MONGO_INSTALL_OPTIONS} ${MONGO_ADDITIONAL_INSTALL_OPTIONS}"
    fi

    helm upgrade --install ${MONGO_HELM_RELEASE_NAME} \
      --version ${MONGO_CHART_VERSION} \
      ${MONGO_INSTALL_OPTIONS} \
      --namespace=${MONGO_NAMESPACE} \
      --create-namespace \
      bitnami/mongodb 1> /dev/null || { echo "Failed to install MongoDB"; return 1; }

    MONGODB_ROOT_PASSWORD=$(kubectl  get secret --namespace ${NAMESPACE} ${MONGO_HELM_RELEASE_NAME}-mongodb -o jsonpath="{.data.mongodb-root-password}" | base64 -d)

    MONGO_URL="mongodb://root:${MONGODB_ROOT_PASSWORD}@${MONGO_HELM_RELEASE_NAME}-mongodb:27017/admin"

    kubectl create secret generic mongodb-url-secret --namespace ${NAMESPACE} --from-literal=password="${MONGO_URL}" 1> /dev/null || { echo "Failed to create MongoDB secret"; return 1; }

  fi

  if [ "${SELF_SIGNED}" = "true" ]; then
    export TLS_CERT_SECRET_NAME=devops-loop-tls-secret
    openssl genrsa -out key.pem 2048
    openssl req -new -x509 -key key.pem -out cert.pem -days 365 \
            -subj "/CN=${DOMAIN}" -addext "subjectAltName = DNS:${DOMAIN},DNS:*.${DOMAIN}" \
            -addext "certificatePolicies = 1.2.3.4"

    kubectl create secret generic ${TLS_CERT_SECRET_NAME} \
      --type=kubernetes.io/tls \
      --from-file=ca.crt=./cert.pem \
      --from-file=tls.crt=./cert.pem \
      --from-file=tls.key=./key.pem \
      --namespace ${NAMESPACE}
  fi

  HELM_OPTIONS="${HELM_OPTIONS:-} \
--set global.domain=${DOMAIN} \
--set-literal global.passwordSeed=${PASSWORD_SEED} \
--set global.platform.smtp.sender=${EMAIL_FROM_ADDRESS} \
--set global.platform.smtp.host=${EMAIL_SERVER_HOST} \
--set global.platform.smtp.port=${EMAIL_SERVER_PORT} \
--set global.platform.smtp.username=${EMAIL_SERVER_USERNAME} \
--set global.platform.smtp.password=${EMAIL_SERVER_PASSWORD} \
--set global.platform.smtp.startTLS=${EMAIL_SERVER_STARTTLS} \
--set global.platform.smtp.smtps=${EMAIL_SERVER_SMTPS} \
--set global.hclCertSecretName=${TLS_CERT_SECRET_NAME} \
--set license=${ACCEPT_LICENSE}
"

  if [ "${SELF_SIGNED}" = "true" ]; then
     HELM_OPTIONS="${HELM_OPTIONS} --set global.privateCaBundleSecretName=${TLS_CERT_SECRET_NAME}"
     HELM_OPTIONS="${HELM_OPTIONS} --set hcl-devops.ingress.cert.selfSigned=true"
  fi

  if [ -n "${LICENSE_SERVER}" ]; then
      HELM_OPTIONS="${HELM_OPTIONS} \
      --set licensing.vendor=HCL \
      --set global.hclLicensingURL=${LICENSE_SERVER} \
      --set global.hclLicensingID=${LICENSE_ID} \
      --set hcl-devops.hclLicensingURL=${LICENSE_SERVER} \
      --set hcl-devops.hclLicensingID=${LICENSE_ID}"
  fi

  if [ -n "${RWX_STORAGE_CLASS}" ]; then
    HELM_OPTIONS="${HELM_OPTIONS} --set global.persistence.rwxStorageClass=${RWX_STORAGE_CLASS}"
  fi

  if [ -n "${RWO_STORAGE_CLASS}" ]; then
    HELM_OPTIONS="${HELM_OPTIONS} --set global.persistence.rwoStorageClass=${RWO_STORAGE_CLASS}"
  fi

  HELM_OPTIONS="${HELM_OPTIONS} ${ADDITIONAL_HELM_OPTIONS}"

  helm upgrade --install ${HELM_NAME} oci://hclcr.io/devops-automation-helm/hcl-devops-loop --version ${LOOP_CHART_VERSION} ${HELM_OPTIONS} \
    -n ${NAMESPACE} || return 1

}
run_install
