package com.mycompany.commerce.seo.url.helpers;

import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.rmi.RemoteException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.ibm.commerce.common.objects.StoreAccessBean;
import com.ibm.commerce.datatype.MapBasedTypedProperty;
import com.ibm.commerce.datatype.TypedProperty;
import com.ibm.commerce.datatype.UrlMapper;
import com.ibm.commerce.datatype.UrlMapperConfig;
import com.ibm.commerce.exception.ParameterNotFoundException;
import com.ibm.commerce.foundation.common.exception.AbstractApplicationException;
import com.ibm.commerce.foundation.server.services.dataaccess.JDBCQueryService;
import com.ibm.commerce.foundation.server.services.dataaccess.exception.DataMediatorException;
import com.ibm.commerce.foundation.server.services.dataaccess.exception.QueryServiceApplicationException;
import com.ibm.commerce.infrastructure.facade.InfrastructureFacadeConstants;
import com.ibm.commerce.registry.StoreRegistry;
import com.ibm.commerce.seo.common.exception.SEOApplicationException;
import com.ibm.commerce.seo.redirect.RedirectEngine;
import com.ibm.commerce.seo.registry.SEOConfigurationRegistry;
import com.ibm.commerce.seo.registry.SEOURLPatternConfigurationRegistry;
import com.ibm.commerce.seo.url.SEOConstants;
import com.ibm.commerce.seo.url.helpers.SEOURLMapper;
import com.ibm.commerce.seo.url.helpers.SEOURLMapperImpl;
import com.ibm.commerce.server.ECConstants;

/**
 * An extended SEOURLMapper to handle performing redirects between legacy URLs
 * and SEO friendly URLs.
 */
public class ExtendedSEOURLMapperImpl extends SEOURLMapperImpl implements
		SEOURLMapper {
  
// Parameter for language ID.
private static final String PARAM_LANGID = "langId";

// Parameter for keyword value.
private static final String PARAM_KEYWORD = "keyword";

// Parameter for the token name.
private static final String PARAM_TOKENNAME = "tokenName";

// Parameter for catalog ID.
private static final String PARAM_CATALOGID = "catalogId";

// Parameter for the token type value of a catalog entry.
private static final String PARAM_CATALOGENTRY = "CatalogEntry";

// Parameter for the token name value of a catalog group.
private static final String PARAM_CATEGORYTOKEN = "CategoryToken";

// Parameter for the token name value of a catalog entry.
private static final String PARAM_PRODUCTTOKEN = "ProductToken";

// Parameter for the token type value of a catalog group.
private static final String PARAM_CATALOGGROUP = "CatalogGroup";

// Parameter for the URL place holder returned from the URL prefix
// which must be removed to do redirects.
private static final String PARAM_URLTOKENPLACEHOLDER = "$SEO:PRIMARYTOKEN$";

//Localhost constant
private static final String LOCALHOST = "localhost";

// Parameter for the URL keyword ID.
private static final String PARAM_KEYWORDID = "keywordId";

// Parameter for the ID of the object, either a catalog entry or catalog
// group.
private static final String PARAM_OBJECTID = "objectId";

// Parameter for the type of the object, which would be a value of either
// CategoryToken or ProductToken.
private static final String PARAM_OBJECTTYPE = "objectType";

// Parameter for the ID of a catalog group.
private static final String PARAM_CATEGORYID = "categoryId";

// Parameter for the ID of a catalog entry in the old style SEO URL.
private static final String PARAM_CATENTRYID = "catentryId";

// Parameter for the ID of a catalog entry in the name value pair type of
// URL.
private static final String PARAM_PRODUCTID = "productId";

// Parameter for store ID.
private static final String PARAM_STOREID = "storeId";

// Name of the URL command called for displaying catalog entries.
private static final String CMD_PRODUCT_DISPLAY = "ProductDisplay";

//Name of the URL command called for displaying catalog entries.
private static final String CMD_PRODUCT_DISPLAY2 = "/Product";

// Name of the URL command called for displaying categories.
private static final String CMD_CATEGORY_DISPLAY = "CategoryDisplay";

//Name of the URL command called for displaying categories.
private static final String CMD_CATEGORY_DISPLAY2 = "/Category";

// Name of the URL command called for displaying top level categories.
private static final String CMD_TOPCATEGORY_DISPLAY = "TopCategoriesDisplay";

// Name of the SQL query in the template file to use for finding the active
// URL keyword value.
private static final String QUERY_ACTIVEKEYWORD = "MyCompany_Select_ActiveKeyword";

// Name of the database column holding the ID of the SEO URL keyword.
private static final String DBCOLUMN_SEOURLKEYWORD_ID = "SEOURLKEYWORD_ID";

// Name of the database column holding the URL keyword value.
private static final String DBCOLUMN_URLKEYWORD = "URLKEYWORD";

// Name of the database column holding the name of the token used for
// the URL keyword.
private static final String DBCOLUMN_TOKENNAME = "TOKENNAME";

// Name of the database column holding the language ID of the token
// used for the URL keyword.
private static final String DBCOLUMN_LANGUAGE_ID = "LANGUAGE_ID";

// Name of the database column holding the ID of the store.
private static final String DBCOLUMN_STOREENT_ID = "STOREENT_ID";

// The separator for where the http protocol text ends.
private static final String PROTOCOL_SEPARATOR = "://";

//The separator for where the http protocol text ends.
private static final String[] PATHS = {CMD_PRODUCT_DISPLAY, CMD_PRODUCT_DISPLAY2, CMD_CATEGORY_DISPLAY, CMD_CATEGORY_DISPLAY2};

//Constant returned by the processRequestURL method to tell the system to 
//treat this URL as an SEO type URL.
private final int URL_TYPE_SEO_PATTERN = 3;

//A map of redirects between non SEO URLs and SEO URLs.
private static Map<String, Map<String, String>> redirects = new HashMap();

/**
 * Check to see if the path in the request is one from the array 
 * of paths that can be redirected.
 * @param path The path in the request.
 * @return True if the path in the request can be redirected, false otherwise.
 */
private boolean isRedirectablePath(String path)
{
	for (int i = 0; i < PATHS.length; i++) {
		if (path.contains(PATHS[i])) {
			return true;
		}
	}
	
	return false;
}


/**
 * Deconstructs an SEO URL into its part to map it to the dynamic URL that
 * it represents. In this extension, redirects are also made between dynamic
 * URLs and SEO URLs.
 * 
 * @param request
 *            The incoming HTTP request.
 * @param response
 *            The outgoing HTTP response.
 * @param storeId
 *            The store ID that this request is made for.
 * @throws SEOApplicationException
 * @return request The modified request object.
 */
public HttpServletRequest deconstructHttpRequest(
		HttpServletRequest request, HttpServletResponse response,
		Integer storeId) throws SEOApplicationException {


	// Perform a redirect if needed.
	String path = request.getPathInfo();
	if (isRedirectablePath(path)) {
		try {
			return performRedirect(request, response, storeId);
		} catch (QueryServiceApplicationException e) {
			e.printStackTrace();
			throw new SEOApplicationException(e);
		} catch (SQLException e) {
			e.printStackTrace();
			throw new SEOApplicationException(e);
		} catch (AbstractApplicationException e) {
			e.printStackTrace();
			throw new SEOApplicationException(e);
		} catch (ParameterNotFoundException e) {
			e.printStackTrace();
			throw new SEOApplicationException(e);
		} catch (NumberFormatException e) {
			e.printStackTrace();
			throw new SEOApplicationException(e);
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
			throw new SEOApplicationException(e);
		}
	}
	else {
		return super.deconstructHttpRequest(request, response, storeId);
	}
	
}

/**
 * Determines if the incoming request is for an SEO URL or not. In this
 * extension a search is performed for any redirects to SEO URLs in the case
 * when this request is not already for an SEO URL.
 * 
 * @param request
 *            The incoming HTTP request.
 * @return List -- Returns an ArrayList with 3 elements
 *         <ul>
 *         <li>Element 0 -- Contains <code>Boolean.TRUE</code> if the
 *         incoming URL is SEO URL based on a pattern. Otherwise, Boolean
 *         FALSE.
 *         <li>Element 1 -- The modified HTTPServletRequest object.
 *         <li>Element 2 -- An Integer object specifying the URL Type, with
 *         the following convention: A value of 0 indicates unknown URL
 *         type. A value of 1 indicates an old SEO URL. A value of 2
 *         indicates a dynamic view/action. A value of 3 indicates a new SEO
 *         URL based on a pattern.
 *         </ul>
 */
public List processRequestURL(HttpServletRequest request) {
     
	// Get the results from the super class method.
	List result = super.processRequestURL(request);
      
	// If this is not an SEO URL find a redirect URL.
	if (result != null && result.size() > 2
		&& !((Boolean) result.get(0)).booleanValue() == true) {
	boolean isURLRedirected = false;
	try {
		try {
			// Find out if this URL is redirected to an SEO URL.
			isURLRedirected = isRedirected(request);
		} catch (ParameterNotFoundException e) {
			e.printStackTrace();
		} catch (NumberFormatException e) {
			e.printStackTrace();
		}
	} catch (QueryServiceApplicationException e) {
		e.printStackTrace();
	} catch (SQLException e) {
		e.printStackTrace();
	} catch (DataMediatorException e) {
		e.printStackTrace();
	}

	// If the URL is redirected then set the response list
	// to say that this is an SEO URL (even though it is not currently)
	// This way it will be treated like an SEO URL so that it will
	// be processed properly later on by the RuntimeServletFilter.
	if (isURLRedirected) {
		result = new ArrayList();
		result.add(true);
		result.add(request);
		result.add(new Integer(3));
	}

	}
	return result;
}

/**
 * Find a redirected URL in the cached redirect map
 * based on the requested URL.
 * @param request The request containing the URL to 
 * search for redirects on.
 * @return The redirected URL if found or null if not found.
 */
public Map<String, String> findRedirectInCache(HttpServletRequest request)
{
	String path = request.getPathInfo();
	String query = request.getQueryString();
	StringBuilder builder = new StringBuilder(path);

	if (query != null) {
		builder.append("?").append(query);
	}
	
	String oldURL = builder.toString();
	
	if (redirects == null)
	{
		redirects = new HashMap();
	}
	return redirects.get(oldURL);
}


/**
 * Determines if a request for a non SEO URL is being redirected to an SEO
 * URL.
 * 
 * @param request
 *            The request to check for redirects on.
 * @return True if the URL in this request is being redirected, false
 *         otherwise.
 * @throws QueryServiceApplicationException
 * @throws SQLException
 * @throws DataMediatorException
 * @throws ParameterNotFoundException
 * @throws NumberFormatException
 */
public boolean isRedirected(HttpServletRequest request)
		throws QueryServiceApplicationException, SQLException,
		DataMediatorException, ParameterNotFoundException,
		NumberFormatException {
 
	String path = request.getPathInfo();
	String objectId = null;
	String objectType = null;
	Map<String, String> keywordInfo = null;
	
	// If this is a category page or a product page
	if (isRedirectablePath(path)) {
		   
		keywordInfo = findRedirectInCache(request);

		if (keywordInfo != null)
		{
			return true;
		}
		
		String query = request.getQueryString();
				
		//Find the store ID in this request.
		String storeId = getParameterFromURL(path, query, PARAM_STOREID);
		
		//If this store is not using the SEO URL Pattern 
		//feature then return false since there is no need to redirect. 
		if (storeId != null && !SEOURLPatternConfigurationRegistry
				.getInstance(request)
				.isSEOURLPatternFeatureEnabled(Integer.parseInt(storeId)))
		{
			return false;
		}
		
		String keywordId = null;
		
		keywordInfo = findActiveKeywordInfo(request);

		if (keywordInfo != null) {
			
			
			addRedirectToCache(request, keywordInfo);
			
			return true;
		}
	}
	// Since no redirect was found return false.
	return false;

}

/**
 * Add the URL from the input request into the cached redirect map
 * pointing to the new URL passed in.
 * @param request The request containing the URL to redirect.
 * @param keywordInfo A map of name value pairs
 * containing properties of the new URL to use.
 */
public void addRedirectToCache(HttpServletRequest request, Map<String, String> keywordInfo)
{
	String path = request.getPathInfo();
	String query = request.getQueryString();
	StringBuilder builder = new StringBuilder(path);

	if (query != null) {
		builder.append("?").append(query);
	}
	
	String oldURL = builder.toString();
	
	if (redirects == null)
	{
		redirects = new HashMap();
	}
	redirects.put(oldURL, keywordInfo);
	
}

/**
 * Perform a 301 redirect between a legacy URL and an SEO friendly URL if a
 * redirect mapping is found for that legacy URL.
 * 
 * @param request
 *            The HTTP request that is being redirected.
 * @param response
 *            The HTTP response to use for the redirect.
 * @param storeId
 *            The store ID that this request is for.
 * @return The response object with the status changed to create a redirect
 *         to the new URL or the original response if no redirect URL is
 *         found.
 * @throws SQLException
 * @throws AbstractApplicationException
 * @throws ParameterNotFoundException
 * @throws NumberFormatException
 * @throws UnsupportedEncodingException 
 */
public HttpServletRequest performRedirect(HttpServletRequest request,
		HttpServletResponse response, Integer storeId) throws SQLException,
		AbstractApplicationException, ParameterNotFoundException,
	NumberFormatException, UnsupportedEncodingException {
	
	Map<String, String> keywordInfo = findRedirectInCache(request);

	// Find the language ID in the request.
	String langId = null;
	String path = request.getPathInfo();
	String query = request.getQueryString();
	
	if (keywordInfo == null)
	{
		// Find the active SEO URL keyword for the object represented by
		// this request.
		keywordInfo = findActiveKeywordInfo(request);
	}

	if (keywordInfo != null) {
	
		String keyword = null;
		String tokenname = null;
		String storeToken = null;
		
		keyword = keywordInfo.get(PARAM_KEYWORD);
		tokenname = keywordInfo.get(PARAM_TOKENNAME);

		// If the language ID was not found in the request
		// parameters then use the language ID of the active
		// keyword (based on what language ID is returned from
		// the findActiveKeywordInfo method)
		if (keywordInfo.get(PARAM_LANGID) != null) {
			langId = keywordInfo.get(PARAM_LANGID);
		}
		else {
			langId = getParameterFromURL(path, query, PARAM_LANGID);
		}
		
		// Find the store Id.
		if (storeId == null) {
			String storeIdString = getParameterFromURL(path, query,
					PARAM_STOREID);
			if (storeIdString != null) {
				storeId = Integer.parseInt(storeIdString);
			}
		}
		
		// Find the catalog ID.
		String catalogId = getParameterFromURL(path, query,
				PARAM_CATALOGID);
		String tokenType = PARAM_CATALOGENTRY;

		// If all the required parameters are found then prepare to
		// create a redirect.
		if (keyword != null && langId != null && catalogId != null
				&& tokenname != null) {
			if (tokenname.equalsIgnoreCase(PARAM_CATEGORYTOKEN)) {
				tokenType = PARAM_CATALOGGROUP;
			}

			TypedProperty urlProperties = new TypedProperty();
			urlProperties.put(ECConstants.EC_STORE_ID, storeId);
			urlProperties.put(ECConstants.EC_CATALOG_ID, catalogId);
			urlProperties.put(ECConstants.EC_LANGUAGE_ID, langId);

			String prefix = null;
			List urlPrefixList = null;
			// Construct the URL prefix for the redirected request.
			// The URL prefix is the part before the actual new SEO URL
			// keyword, such as the context root.
			try {
				urlPrefixList = SEOConfigurationRegistry.singleton()
				.getMapperInstance()
				.constructSEOUrlPrefixByUsageAndDevice(tokenType,
						SEOConstants.DEVICE_TYPE_BROWSER,
						urlProperties);
			}
			catch(NullPointerException e) {
				e.printStackTrace();
			}
			
			if (urlPrefixList != null && urlPrefixList.size() > 0) {
				prefix = String.valueOf(urlPrefixList.get(0));
			}

			if (prefix != null && keyword != null) {
				String fullURL = prefix;

				fullURL = fullURL.replace(PARAM_URLTOKENPLACEHOLDER,
						keyword);
				
				String requestURL = request.getRequestURL().toString();
				String hostname = null;
				try {
					URL url = new URL(requestURL);
					hostname = url.getHost();
				} catch (MalformedURLException e) {
					e.printStackTrace();
				}
				
				if (hostname != null)
				{
					fullURL = fullURL.replaceFirst(LOCALHOST, hostname);
				}
				
				RedirectEngine.performRedirect(response, fullURL);
			}
			
			if (path.contains(CMD_PRODUCT_DISPLAY2)
					|| path.contains(CMD_CATEGORY_DISPLAY2)) {
				
				// Instantiate the class used to decipher the old style SEO URLs.
				UrlMapper mapper = UrlMapperConfig.getUrlMapper();
				
				if (mapper != null) {
					request = mapper
							.createMappedWrapperRequest(request);
				}
			}
		}
	}
	
	return request;
}

/**
 * Find information such as ID and type of object for a catalog entry or
 * catalog group.
 * 
 * @param path
 *            The URL path for this request.
 * @param query
 *            The query parameters for this request if applicable.
 * @return The ID and object type (either ProductToken or CategoryToken) of
 *         the catalog entry or catalog group that this request is for.
 */
protected Map findObjectInfo(String path, String query) {
	String objectID = null;
	String objectType = null;
	Map response = new HashMap();

	if (path.contains(CMD_PRODUCT_DISPLAY)) {
		// If the object in this request is a catalog entry and the request
		// is using the name value pair style URL
		objectID = getParameterFromURL(path, query, PARAM_PRODUCTID);
		objectType = PARAM_PRODUCTTOKEN;

	} else if (path.contains(CMD_CATEGORY_DISPLAY)) {
		// If the object in this request is a catalog group and the request
		// is using the name value pair style URL
		objectID = getParameterFromURL(path, query, PARAM_CATEGORYID);
		objectType = PARAM_CATEGORYTOKEN;
	} else {
		// If the object in this request is a catalog entry and the request
		// is using the old style SEO URL
		objectID = getParameterFromURL(path, query, PARAM_CATENTRYID);
		objectType = PARAM_PRODUCTTOKEN;
		if (objectID == null) {
			// If the object in this request is a catalog group and the
			// request is using the old style SEO URL
			objectID = getParameterFromURL(path, query, PARAM_CATEGORYID);
			objectType = PARAM_CATEGORYTOKEN;
		}
	}

	response.put(PARAM_OBJECTID, objectID);
	response.put(PARAM_OBJECTTYPE, objectType);

	return response;
}

/**
 * Find the active SEO URL keyword for an object with a specified ID (either
 * a catalog entry or catalog group).
 * 
 * @param objectId
 *            The ID of the catalog entry or catalog group.
 * @param objectType
 *            The type of object being acted on, either ProductToken or
 *            CategoryToken.
 * @param request
 *            The request being acted on.
 * @return A map of values for the active keyword of this catalog entry or
 *         catalog group. The parameters include: keywordId: The ID of the
 *         SEO URL keyword. keyword: The actual SEO URL keyword text.
 *         tokenName: The type of keyword this is, either CatalogEntry or
 *         CatalogGroup. langId: The language ID of the keyword.
 * @throws ParameterNotFoundException
 * @throws NumberFormatException
 * @throws QueryServiceApplicationException
 * @throws SQLException
 */
protected Map<String, String> findActiveKeywordInfoOld(String objectId,
		String objectType, HttpServletRequest request)
		throws ParameterNotFoundException, NumberFormatException,
		QueryServiceApplicationException, SQLException {
	String path = request.getPathInfo();
	String query = request.getQueryString();

	// Find the store ID in the request.
	String storeId = getParameterFromURL(path, query, PARAM_STOREID);

	// Find the language ID in the request.
	String langId = getParameterFromURL(path, query, PARAM_LANGID);

	if (storeId != null && langId == null) {
		// If the language ID could not be found in the request then look
		// for it in the header of the request.
		langId = request.getParameter(PARAM_LANGID);

		if (langId == null) {
			// If the language ID cannot be found in the request at all then
			// find the store default language ID from the StoreAccessBean.
			StoreAccessBean store = StoreRegistry.singleton().getStore(
					Integer.parseInt(storeId));
			try {
				langId = store.getLanguageId();
			} catch (RemoteException e) {
				e.printStackTrace();
			} catch (CreateException e) {
				e.printStackTrace();
			} catch (FinderException e) {
				e.printStackTrace();
			} catch (NamingException e) {
				e.printStackTrace();
			}
		}
	}

	Map keywordInfo = new HashMap();
	
	if (storeId != null && langId != null) {
		Map params = new HashMap();
		addParameter(params, PARAM_STOREID, storeId);
		addParameter(params, PARAM_LANGID, langId);
		addParameter(params, PARAM_OBJECTID, objectId);
		addParameter(params, PARAM_TOKENNAME, objectType);
	
		// Query the database to find the active keyword for the object in
		// this request.
		List<Map> results = executeQuery(QUERY_ACTIVEKEYWORD, params);
		Iterator it = results.iterator();
	
		// Find the active keyword info for the e-site if it overrides the
		// asset store for this object.
		// Otherwise return the active keyword info from the asset store.
		while (it.hasNext()) {
			keywordInfo.clear();
	
			Map<String, Object> row = (Map) it.next();
	
			String storeIdFromDB = String.valueOf(row
					.get(DBCOLUMN_STOREENT_ID));
	
			keywordInfo.put(PARAM_KEYWORDID, String.valueOf(row
					.get(DBCOLUMN_SEOURLKEYWORD_ID)));
			keywordInfo.put(PARAM_KEYWORD, String.valueOf(row
					.get(DBCOLUMN_URLKEYWORD)));
			keywordInfo.put(PARAM_TOKENNAME, String.valueOf(row
					.get(DBCOLUMN_TOKENNAME)));
			keywordInfo.put(PARAM_LANGID, String.valueOf(row
					.get(DBCOLUMN_LANGUAGE_ID)));
	
			if (storeId.equals(storeIdFromDB)) {
				break;
			}
		}
	
		// Return the keyword if it is found.
		return keywordInfo;
	}

	return null;
}




/**
 * Find the active SEO URL keyword for an object with a specified ID (either
 * a catalog entry or catalog group).
 * 
 * @param objectId
 *            The ID of the catalog entry or catalog group.
 * @param objectType
 *            The type of object being acted on, either ProductToken or
 *            CategoryToken.
 * @param request
 *            The request being acted on.
 * @return A map of values for the active keyword of this catalog entry or
 *         catalog group. The parameters include: keywordId: The ID of the
 *         SEO URL keyword. keyword: The actual SEO URL keyword text.
 *         tokenName: The type of keyword this is, either CatalogEntry or
 *         CatalogGroup. langId: The language ID of the keyword.
 * @throws ParameterNotFoundException
 * @throws NumberFormatException
 * @throws QueryServiceApplicationException
 * @throws SQLException
 */
@SuppressWarnings("null")
protected Map<String, String> findActiveKeywordInfo(HttpServletRequest request)
		throws ParameterNotFoundException, NumberFormatException,
		QueryServiceApplicationException, SQLException {
	
	String path = request.getPathInfo();
	String query = request.getQueryString();
	String objectId = null;
	String objectType = null;
	
	Map<String, String> objectInfo = findObjectInfo(path, query);
	
	if (objectInfo != null) {
		objectId = objectInfo.get(PARAM_OBJECTID);
		objectType = objectInfo.get(PARAM_OBJECTTYPE);
	}
	
	if (objectId != null)
	{
		// Find the store ID in the request.
		String storeId = getParameterFromURL(path, query, PARAM_STOREID);

		// Find the language ID in the request.
		String langId = getParameterFromURL(path, query, PARAM_LANGID);
		Map keywordInfo = null;
		
		if (langId == null) {
			
			// If the language ID could not be found in the request then look
			// for it in the header of the request.
			langId = request.getParameter(PARAM_LANGID);
			
			if (langId == null){
				// If the language ID cannot be found in the request at all then
				// find the store default language ID from the StoreAccessBean.
				StoreAccessBean store = StoreRegistry.singleton().getStore(
						Integer.parseInt(storeId));
				try {
					langId = store.getLanguageId();
				} catch (RemoteException e) {
					e.printStackTrace();
				} catch (CreateException e) {
					e.printStackTrace();
				} catch (FinderException e) {
					e.printStackTrace();
				} catch (NamingException e) {
					e.printStackTrace();
				}
			}
		}
		
		if (storeId != null && langId != null) {
			
			Map params = new HashMap();
			addParameter(params, PARAM_STOREID, storeId);
			addParameter(params, PARAM_LANGID, langId);
			addParameter(params, PARAM_OBJECTID, objectId);
			addParameter(params, PARAM_TOKENNAME, objectType);
		
			// Query the database to find the active keyword for the object in
			// this request.
			List<Map> results = executeQuery(QUERY_ACTIVEKEYWORD, params);
			Iterator it = results.iterator();
		
			if (it.hasNext())
			{
				keywordInfo = new HashMap();
			}
			// Find the active keyword info for the e-site if it overrides the
			// asset store for this object.
			// Otherwise return the active keyword info from the asset store.
			while (it.hasNext()) {
				keywordInfo.clear();
		
				Map<String, Object> row = (Map) it.next();
		
				String storeIdFromDB = String.valueOf(row
						.get(DBCOLUMN_STOREENT_ID));
		
				keywordInfo.put(PARAM_KEYWORDID, String.valueOf(row
						.get(DBCOLUMN_SEOURLKEYWORD_ID)));
				keywordInfo.put(PARAM_KEYWORD, String.valueOf(row
						.get(DBCOLUMN_URLKEYWORD)));
				keywordInfo.put(PARAM_TOKENNAME, String.valueOf(row
						.get(DBCOLUMN_TOKENNAME)));
				keywordInfo.put(PARAM_LANGID, String.valueOf(row
						.get(DBCOLUMN_LANGUAGE_ID)));
		
				if (storeId.equals(storeIdFromDB)) {
					break;
				}
			}
		
			// Return the keyword if it is found.
			return keywordInfo;
			
		}

		
		
		
	}
	
	return null;
}

/**
 * Find the value of a specified parameter in a URL.
 * 
 * @param path
 *            The URL path for this request.
 * @param query
 *            The query parameters for this request if applicable.
 * @param parameterName
 *            The name of the parameter to find.
 * @return The value of the parameter if found or null otherwise.
 */
protected String getParameterFromURL(String path, String query,
		String parameterName) {

		// Instantiate the class used to decipher the old style SEO URLs.
		UrlMapper mapper = UrlMapperConfig.getUrlMapper();

		String parameterValue = null;

		if (mapper != null) {
			String[] parts = path.split(mapper.getParamSeparator());

			// If this request is for an old style SEO URL.
			if (parts.length > 1) {
				String reqId = parts[0];
				reqId = reqId.replace("/", "");
				HashMap map = new HashMap();

				TypedProperty prop = new MapBasedTypedProperty(map);

				if (reqId != null) {

					// Find the desired parameter value using the URLMapper.
					mapper.extractRequestParameters(reqId, path, prop);
					Map temp = prop.toMap();

					if (temp.get(parameterName) != null) {
						String[] tempArray = (String[]) temp.get(parameterName);
						parameterValue = tempArray[0];
					}
				}
			} else {
				// If this request is for a name value pair style URL then find
				// the desired parameter in the query string of the request.
				Map parameters = extractQueryParametersAsMap(query);

				if (parameters != null && parameters.get(parameterName) != null) {
					parameterValue = (String) parameters.get(parameterName);
				}
			}
		}

		return parameterValue;
}

/**
 * Executes a query against the infrastructure component using the
 * JDBCQueryService. This method will also clear out all the parameters in
 * the input map so that the same map can be used fresh in a future query.
 * 
 * @param queryName
 *            The name of the query to execute.
 * @param params
 *            A map of parameters to use in the query. Each key in the map
 *            is the name of the parameter and each value is a list of
 *            corresponding values for that parameter.
 * @return A list of resulting rows with each entry in the list being a map
 *         of name value pairs representing column name and value pairs from
 *         the table.
 * @throws QueryServiceApplicationException
 * @throws SQLException
 */
public static List executeQuery(String queryName, Map params)
		throws QueryServiceApplicationException, SQLException {
	
	// Executes a query against the database using the JDBCQueryService.
	JDBCQueryService query = new JDBCQueryService(
			InfrastructureFacadeConstants.COMPONENT_NAME);
	List results = query.executeQuery(queryName, params);
	params.clear();
	return results;
}

/**
 * Adds a parameter into a map in the correct format for use in the
 * executeQuery method, which is a name to list of values.
 * 
 * @see executeQuery.
 * @param params
 *            The map to add the parameter to.
 * @param paramName
 *            The name of the parameter to add.
 * @param paramValue
 *            The value to add to the map.
 */
public static void addParameter(Map params, String paramName,
		String paramValue) {
	
	List paramValues = new ArrayList();
	paramValues.add(paramValue);
	params.put(paramName, paramValues);
}

/**
 * Extract parameters from a query string and return them as a map with the
 * keys being the names of the query parameters and the values in the map
 * being the corresponding values of those query parameters.
 * 
 * @param query
 *            A query string containing name value pairs like
 *            storeId=10101&catalogId=10020&catentryId=89383
 * @return A map of query parameters.
 */
private Map extractQueryParametersAsMap(String query) {
	Map params = new HashMap();
	String[] queryParameters = query.split("&");

	for (int i = 0; i < queryParameters.length; i++) {
		if (queryParameters[i].split("=").length == 2)
		{
			String key = queryParameters[i].split("=")[0];
			String value = queryParameters[i].split("=")[1];
			params.put(key, value);
		}
		
	}
	return params;
}

/**
 * Encode a set of keywords in UTF-8 format. This will leave the separation
 * character and protocol intact.
 * 
 * @param url
 *            The URL to encode.
 * @return The URL with all keywords encoded
 * @throws UnsupportedEncodingException
 */
public static String encodeURL(String url)
		throws UnsupportedEncodingException {
		
		String protocol = "";
		boolean trailingSlash = false;
		if (SEOConstants.FORWARD_SLASH.equals(String.valueOf(url.charAt(url
				.length() - 1)))) {
			trailingSlash = true;
		}
		StringBuilder encodedURL = new StringBuilder();
		if (url.indexOf(PROTOCOL_SEPARATOR) != -1) {
			protocol = url.split(PROTOCOL_SEPARATOR)[0];
			url = url.split(PROTOCOL_SEPARATOR)[1];
		}

		String[] keywords = url.split(SEOConstants.FORWARD_SLASH);
		StringBuilder newURL = new StringBuilder();

		for (int i = 0; i < keywords.length; i++) {
			String encodedKeyword = URLEncoder.encode(keywords[i], "UTF-8");
			newURL.append(encodedKeyword);

			if (i < keywords.length - 1 || trailingSlash) {
				newURL.append(SEOConstants.FORWARD_SLASH);
			}
		}

		if (!protocol.equals("")) {
			encodedURL.append(protocol).append(PROTOCOL_SEPARATOR);
		}
		encodedURL.append(newURL);

		return encodedURL.toString();
}

}
