/*
 *==================================================
 * Licensed Materials - Property of HCL Technologies
 *
 * HCL Commerce
 *
 * (C) Copyright HCL Technologies Limited 2021
 *
 *==================================================
 */

import React, { useMemo } from "react";
//UI
import {
  StyledGrid,
  StyledPDPContainer,
  StyledTypography,
  StyledButton,
  StyledTabs,
} from "@hcl-commerce-store-sdk/react-component";
import { ProductAttributes } from "@hcl-commerce-store-sdk/react-component";
import { ProductImage } from "@hcl-commerce-store-sdk/react-component";
import { ProductQuantity } from "@hcl-commerce-store-sdk/react-component";
import { ProductThumbnails } from "@hcl-commerce-store-sdk/react-component";
import Hidden from "@material-ui/core/Hidden";
import { get } from "lodash-es";

interface ProductDetailsWidgetProps {
  productPartNumber: any;
  product: any;
  showCarousel: boolean;
  carouselImages: any;
  changeMainImage: Function;
  index: number;
  displayFullImage: string;
  displayName: string;
  displayPartNumber: string;
  displayShortDesc: string;
  promotion: any;
  displayOfferPrice: number;
  displayListPrice: number;
  definingAttrList: any;
  skuColor: string;
  onAttributeChange: Function;
  currentSelection: any;
  updateProductQuantity: Function;
  availability: any;
  addToCart: Function;
  inventoryAvailableFlag: boolean;
  buyableFlag: boolean;
  productDetailTabsChildren: any;
  translation: any;
  formattedPriceDisplayOffer: any;
  formattedPriceDisplayList: any;
  formattedPriceDisplayNull: any;
  loginStatus: boolean;
  userWishList: any;
  addToWishList: Function;
}

export const ProductDetailsWidget: React.FC<ProductDetailsWidgetProps> = (
  props: any
) => {
  const {
    productPartNumber,
    product,
    showCarousel,
    carouselImages,
    changeMainImage,
    index,
    displayFullImage,
    displayName,
    displayPartNumber,
    displayShortDesc,
    promotion,
    displayOfferPrice,
    displayListPrice,
    definingAttrList,
    skuColor,
    onAttributeChange,
    currentSelection,
    updateProductQuantity,
    availability,
    addToCart,
    inventoryAvailableFlag,
    buyableFlag,
    productDetailTabsChildren,
    translation,
    formattedPriceDisplayOffer,
    formattedPriceDisplayList,
    formattedPriceDisplayNull,
    loginStatus,
    userWishList,
    addToWishList,
  } = props;

  // for each sku, an object that represents its attributes, e.g.,
  // [
  //   sku1, --> { Color: "blue", Size: "48x48x48" },
  //   sku2  --> { Color: "red",  Size: "52x52x52" }
  // ]
  const skusAsAttrs = useMemo(() => {
    return (product?.items ?? []).map((p) => {
      const rc = {};
      p.attributes
        ?.filter(({ usage: u }) => u === "Defining")
        .forEach((a) => (rc[a.identifier] = get(a, "values[0].identifier")));
      return rc;
    });
  }, [product]);

  return (
    <>
      {productPartNumber && product && (
        <StyledPDPContainer
          itemScope
          itemType="http://schema.org/Product"
          id={`product-image-details_${productPartNumber}`}>
          <StyledGrid container>
            <StyledGrid container spacing={2} item xs={12}>
              {showCarousel ? (
                <>
                  <StyledGrid item xs={12} md={1}>
                    <ProductThumbnails
                      imageList={carouselImages}
                      changeMainImage={changeMainImage}
                      index={index}
                    />
                  </StyledGrid>
                  <Hidden smDown>
                    <StyledGrid item xs={5} className="product-image">
                      <ProductImage
                        fullImage={displayFullImage}
                        isAngleImage={false}
                        alt={displayName}
                      />
                    </StyledGrid>
                  </Hidden>
                </>
              ) : (
                <StyledGrid item xs={12} md={6} className="product-image">
                  <ProductImage
                    fullImage={displayFullImage}
                    isAngleImage={false}
                    alt={displayName}
                  />
                </StyledGrid>
              )}
              <StyledGrid item xs={12} sm={6} lg={5}>
                {displayName && (
                  <StyledTypography
                    variant="h4"
                    itemProp="name"
                    className="product-name">
                    {displayName}
                  </StyledTypography>
                )}
                {displayPartNumber && (
                  <StyledTypography variant="body2" className="product-sku">
                    {translation.productDetailSKU}: {displayPartNumber}
                  </StyledTypography>
                )}
                {displayShortDesc && (
                  <StyledTypography
                    variant="body1"
                    itemProp="description"
                    className="product-shortDescription">
                    {displayShortDesc}
                  </StyledTypography>
                )}
                {promotion && (
                  <StyledTypography
                    variant="body2"
                    id={`product_advertisement_${productPartNumber}`}
                    className="product-promo"
                    gutterBottom>
                    {promotion}
                  </StyledTypography>
                )}
                <div
                  itemProp="offers"
                  itemScope
                  itemType="http://schema.org/Offer">
                  {product.type !== "bundle" && (
                    <>
                      <StyledTypography
                        variant="h5"
                        className="product-price-container">
                        {displayOfferPrice > 0 && (
                          <span className="product-price">
                            {formattedPriceDisplayOffer}
                          </span>
                        )}
                        {displayOfferPrice > 0 && displayListPrice > 0 && (
                          <span
                            id={`product_price_${productPartNumber}`}
                            className={
                              displayListPrice > 0 ? "strikethrough" : ""
                            }>
                            {formattedPriceDisplayList}
                          </span>
                        )}
                        {displayOfferPrice === 0 && (
                          <span id={`product_offer_price_${productPartNumber}`}>
                            {formattedPriceDisplayNull}
                          </span>
                        )}
                      </StyledTypography>
                      {definingAttrList?.length > 0 && (
                        <ProductAttributes
                          skusAsAttrs={skusAsAttrs}
                          skuColor={skuColor}
                          attributeList={definingAttrList}
                          onChangeHandler={onAttributeChange}
                          currentSelection={currentSelection}
                        />
                      )}
                      <>
                        <StyledTypography
                          variant="body2"
                          className="product-quantity">
                          {translation.productDetailQuantity}
                        </StyledTypography>
                        <ProductQuantity
                          updateProductQuantity={updateProductQuantity}
                          value={currentSelection.quantity}
                        />
                        {availability?.length > 0 && (
                          <>
                            <StyledTypography
                              variant="body2"
                              className="product-availability">
                              {translation.productDetailAvailability}
                            </StyledTypography>
                            <StyledTypography variant="body1" component="div">
                              {availability.map((e: any) => (
                                <div key={`inventoryDetails_div_${e.storeId}`}>
                                  <div
                                    key={`store-name_div_${e.storeId}`}
                                    id={`product_availability_store_name_${productPartNumber}`}
                                    className="store-name">
                                    {e.storeName}
                                  </div>
                                  {e.inventoryStatus && (
                                    <div
                                      key={`inStock_div_${e.storeId}`}
                                      className="inventory-status in-stock"
                                      id={`product_availability_status_inStock_${productPartNumber}`}>
                                      {
                                        translation.CommerceEnvironmentinventoryStatusAvailable
                                      }
                                    </div>
                                  )}
                                  {!e.inventoryStatus && (
                                    <div
                                      key={`outOfStock_div_${e.storeId}`}
                                      className="store-inventory out-of-stock"
                                      id={`product_availability_status_outOfStock_${productPartNumber}`}>
                                      {
                                        translation.CommerceEnvironmentinventoryStatusOOS
                                      }
                                    </div>
                                  )}
                                </div>
                              ))}
                            </StyledTypography>
                          </>
                        )}
                        <StyledButton
                          color="primary"
                          size="small"
                          className="product-add-to-cart"
                          id={`product_add_to_cart_${productPartNumber}`}
                          onClick={addToCart}
                          disabled={
                            !inventoryAvailableFlag ||
                            !buyableFlag ||
                            !displayOfferPrice
                          }>
                          {translation.productDetailAddToCart}
                        </StyledButton>
                        {loginStatus && (
                          <StyledButton
                            color="secondary"
                            size="small"
                            className="left-margin-2"
                            disabled={!(loginStatus && userWishList)}
                            onClick={addToWishList}
                            id={`product_add_to_cart_${productPartNumber}`}>
                            Add to Wish List
                          </StyledButton>
                        )}
                      </>
                    </>
                  )}
                </div>
              </StyledGrid>
            </StyledGrid>
            <StyledGrid item xs={12}>
              {productDetailTabsChildren?.length > 0 && (
                <StyledTabs
                  key={currentSelection.sku.partNumber}
                  // adding key here to make sure tab is full re-rendered
                  //with new local state value according to different sku
                  childrenList={productDetailTabsChildren}
                  name="productDetails"
                />
              )}
            </StyledGrid>
          </StyledGrid>
        </StyledPDPContainer>
      )}
    </>
  );
};
