package com.mycompany.commerce.project.facade.server.services.dataaccess.bom.mediator;

/*
 *-----------------------------------------------------------------
 * IBM Confidential
 *
 * OCO Source Materials
 *
 * WebSphere Commerce
 *
 * (C) Copyright IBM Corp. 2008
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has
 * been deposited with the U.S. Copyright Office.
 *-----------------------------------------------------------------
 */

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.ibm.commerce.foundation.client.util.oagis.RelationalExpression;
import com.ibm.commerce.foundation.common.datatypes.CommerceFoundationFactory;
import com.ibm.commerce.foundation.common.datatypes.DescriptionType;
import com.ibm.commerce.foundation.common.exception.AbstractApplicationException;
import com.ibm.commerce.foundation.common.exception.ApplicationError;
import com.ibm.commerce.foundation.common.util.logging.LoggingHelper;
import com.ibm.commerce.foundation.server.services.dataaccess.PhysicalDataContainer;
import com.ibm.commerce.foundation.server.services.dataaccess.SelectionCriteria;
import com.ibm.commerce.foundation.server.services.dataaccess.bom.exception.BusinessObjectMediatorException;
import com.ibm.commerce.foundation.server.services.dataaccess.bom.mediator.AbstractChangeBusinessObjectMediatorImpl;
import com.ibm.commerce.foundation.server.services.dataaccess.exception.DataMediatorException;
import com.mycompany.commerce.project.facade.ProjectFacadeConstants;
import com.mycompany.commerce.project.facade.datatypes.ProjectCollectionType;
import com.mycompany.commerce.project.facade.server.entity.datatypes.Xprjcol;
import com.mycompany.commerce.project.facade.server.helpers.ProjectComponentHelper;
import com.mycompany.commerce.project.facade.server.metadata.ProjectMetadata;
import com.mycompany.commerce.project.logging.ProjectApplicationMessageKeys;

/**
 * Add and Delete operation on the logical noun for ProjectCollection
 */
public class ChangeProjectCollectionMediator extends
		AbstractChangeBusinessObjectMediatorImpl {

	private static final String CLASSNAME = ChangeProjectCollectionMediator.class
			.getName();

	private static final Logger LOGGER = LoggingHelper
			.getLogger(ChangeProjectCollectionMediator.class);

	/**
	 * Creates the physical entity corresponding to the logical noun
	 * 
	 * @param aNoun
	 *            (ProjectCollectionType) The logical business object for which
	 *            the physical entity needs to be created
	 * @return The entity container for physical objects
	 * @throws AbstractApplicationException
	 */
	public void create(Object aNoun) throws AbstractApplicationException {

		final String METHODNAME = "create";
		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.entering(CLASSNAME, METHODNAME, new Object[] { aNoun });
		}

		// Create the entity
		ProjectCollectionType logicalProjectCollection = (ProjectCollectionType) aNoun;
		PhysicalDataContainer pdc = getPhysicalDataContainer();

		Xprjcol physicalProjectCollection = (Xprjcol) pdc.createPhysicalObject(
				null, Xprjcol.class);

		// get thd next available uid and then insert into the physical
		// dataobject
		Long uid = pdc.getNextPrimaryKey(Xprjcol.class);
		physicalProjectCollection.setXprjcol_id(uid.intValue());

		logicalProjectCollection.getProjectCollectionIdentifier().setUniqueID(
				Long.toString(uid.longValue()));
		if (LoggingHelper.isTraceEnabled(LOGGER)) {
			LOGGER.logp(Level.FINE, CLASSNAME, METHODNAME,
					"new ProjectCollection unique id= " + uid);
		}

		Integer storeId = null;
		if (logicalProjectCollection.getProjectCollectionIdentifier()
				.getExternalIdentifier().getStoreIdentifier() != null
				&& logicalProjectCollection.getProjectCollectionIdentifier()
						.getExternalIdentifier().getStoreIdentifier()
						.getUniqueID() != null
				&& logicalProjectCollection.getProjectCollectionIdentifier()
						.getExternalIdentifier().getStoreIdentifier()
						.getUniqueID().length() > 0) {
			//Get store id from the noun
			storeId = Integer.valueOf(logicalProjectCollection
					.getProjectCollectionIdentifier().getExternalIdentifier()
					.getStoreIdentifier().getUniqueID());
		} else {
			//Get store id from the context
			storeId = ProjectComponentHelper.getStoreId();
		}

		// set the name
		if (logicalProjectCollection.getProjectCollectionIdentifier() != null
				&& logicalProjectCollection.getProjectCollectionIdentifier()
						.getExternalIdentifier() != null) {
			physicalProjectCollection.setPrjcolname(logicalProjectCollection
					.getProjectCollectionIdentifier().getExternalIdentifier()
					.getName());
			physicalProjectCollection.setStoreent_id(storeId.intValue());
		}

		createNounParts(aNoun, physicalProjectCollection);		

		// update the user data field
		updateNameValuePairType(logicalProjectCollection,
				physicalProjectCollection);

		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.exiting(CLASSNAME, METHODNAME);
		}
	}

	/**
	 * For each noun part present in the project noun, this method updates the
	 * physical catentry.
	 * 
	 * @param aNoun
	 *            The logical project noun.
	 * @param aPhysicalEntity
	 *            The physical project to be updated.
	 * @throws AbstractApplicationException
	 */
	private void createNounParts(Object aNoun, Object aPhysicalEntity)
			throws AbstractApplicationException {
		// create the noun parts by invoking the part mediators
		ProjectCollectionType aNounType = (ProjectCollectionType) aNoun;

		if (aNounType.getDescription().size() > 0) {
			Iterator anIter = aNounType.getDescription().iterator();
			while (anIter.hasNext()) {
				DescriptionType aDescription = (DescriptionType) anIter.next();
				createNounPart(aNoun, aDescription, aPhysicalEntity);

			}
		}
	}

	/**
	 * Should not be called
	 * 
	 * @param aNoun
	 *            (ProjectCollectionType) The logical noun with updated fields
	 * @throws BusinessObjectMediatorException
	 */
	public void change(Object aNoun) throws BusinessObjectMediatorException {
		final String METHODNAME = "change";
		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.entering(CLASSNAME, METHODNAME, new Object[] { aNoun });
			LOGGER.exiting(CLASSNAME, METHODNAME);
		}
	}

	/**
	 * Deletes the logical noun. The delete action on the logical noun is copied
	 * to the physical implementation. Deletion of logical noun may result in
	 * update of physical entity in the form of mark for delete or may result in
	 * an actual delete from physical data store
	 * 
	 * @param aNoun
	 *            (ProjectCollectionType) The logical noun that should be
	 *            deleted
	 * @throws BusinessObjectMediatorException
	 * @throws DataMediatorException
	 */
	public void delete(Object aNoun) throws BusinessObjectMediatorException,
			DataMediatorException {

		final String METHODNAME = "delete";
		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.entering(CLASSNAME, METHODNAME, new Object[] { aNoun });
		}

		// Delete the entity; can either mean mark for delete, or delete

		ProjectCollectionType logicalProjectCollection = (ProjectCollectionType) aNoun;
		Integer projectcollectionId = Integer.valueOf(logicalProjectCollection
				.getProjectCollectionIdentifier().getUniqueID());

		// Get the physical data object with specific id
		Xprjcol physicalProjectCollection = (Xprjcol) findPhysicalEntity(aNoun);		

		if (physicalProjectCollection != null) {
			if (LoggingHelper.isTraceEnabled(LOGGER)) {
				LOGGER.logp(Level.FINE, CLASSNAME, METHODNAME,
						"Delete Data Object ProjectCollection with unqiue ID: "
								+ projectcollectionId);
			}
			getPhysicalDataContainer().removePhysicalObject(
					physicalProjectCollection);
		} else {
			if (LoggingHelper.isTraceEnabled(LOGGER)) {
				LOGGER.logp(Level.FINE, CLASSNAME, METHODNAME,
						"Data Object ProjectCollection with unqiue ID: "
								+ projectcollectionId + " doesn't exist");
			}
		}

		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.exiting(CLASSNAME, METHODNAME);
		}
	}

	/**
	 * Find physical entity in the physical data container
	 * 
	 * @param aNoun
	 *            The logical <code><ProjectCollectionType</code> SDO.
	 * @return This method returns the <code>Xprjcol</code> object
	 *         if it is found, otherwise it returns null.
	 * @throws DataMediatorException
	 */
	public Object findPhysicalEntity(Object aNoun) throws DataMediatorException {

		final String METHODNAME = "findPhysicalEntity";
		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.entering(CLASSNAME, METHODNAME, new Object[] { aNoun });
		}

		Xprjcol aPhysicalEntity = null;
		PhysicalDataContainer pdc = getPhysicalDataContainer();

		// if the PhysicalDataContainer is not null, loop it and compare the uid
		// to find the DataObject
		if (pdc != null) {
			ProjectCollectionType aLogicalProjectCollection = (ProjectCollectionType) aNoun;

			if (aLogicalProjectCollection.getProjectCollectionIdentifier()
					.getUniqueID() != null) {
				long projectcollectionId = Long
						.parseLong(aLogicalProjectCollection
								.getProjectCollectionIdentifier().getUniqueID());
				if (LoggingHelper.isTraceEnabled(LOGGER)) {
					LOGGER.logp(Level.FINE, CLASSNAME, METHODNAME,
							"Find ProjectCollection: " + projectcollectionId);
				}

				List projectcollectionList = pdc.getPhysicalObjects();
				if (projectcollectionList != null
						&& projectcollectionList.size() > 0) {
					Iterator projectcollectionIter = projectcollectionList
							.iterator();
					while (projectcollectionIter.hasNext()) {
						Xprjcol tempPhysicalEntity = (Xprjcol) projectcollectionIter
								.next();
						// compare id to get the Physical DataObject
						if (tempPhysicalEntity.getXprjcol_id() == projectcollectionId) {
							
							aPhysicalEntity = tempPhysicalEntity;
							break;
						}
					}
				}

			}
		}

		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.exiting(CLASSNAME, METHODNAME, aPhysicalEntity);
		}

		return aPhysicalEntity;

	}

	/**
	 * Initialize the physical data container for this mediator. Obtain the
	 * physical entities that correspond to this logical entity. The data from
	 * noun present in the list will be used to fetch the physical entities with
	 * the help of queries configured with DSL
	 * 
	 * @throws DataMediatorException
	 * @see com.ibm.commerce.foundation.server.services.dataaccess.bom.mediator.AbstractChangeBusinessObjectMediatorImpl#initializePhysicalDataContainer(java.util.List)
	 */
	protected PhysicalDataContainer initializePhysicalDataContainer(
			List aListNoun) throws DataMediatorException {

		final String METHODNAME = "initializePhysicalDataContainer";
		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.entering(CLASSNAME, METHODNAME, new Object[] { aListNoun });
		}

		PhysicalDataContainer physicalEntities = null;

		// Get a list of extenal IDs from the nouns
		List projectcollectionIdList = new ArrayList();
		Iterator projectcollectionNounIterator = aListNoun.iterator();
		while (projectcollectionNounIterator.hasNext()) {
			ProjectCollectionType aProjectCollectionType = (ProjectCollectionType) projectcollectionNounIterator
					.next();
			if (aProjectCollectionType.getProjectCollectionIdentifier() != null) {
				String uniqueID = aProjectCollectionType
						.getProjectCollectionIdentifier().getUniqueID();
				if (uniqueID != null && uniqueID.trim().length() > 0) {
					projectcollectionIdList.add(uniqueID);
				}
			}
		}

		if (projectcollectionIdList.size() > 0) {
			// if there is one or more noun, construct the query to get the
			// physical data containers
			SelectionCriteria query = new SelectionCriteria();
			query
					.setXPathKey(ProjectFacadeConstants.XPATH_PROJECTCOLLECTION_UID
							+ ProjectFacadeConstants.DOUBLE_CLOSE_XPATH);
			query
					.setAccessProfile(ProjectFacadeConstants.ACCESS_PROFILE_UPDATE);
			query
					.setXPathParameter(new RelationalExpression(
							ProjectFacadeConstants.NOUN_ATTRIBUTE_NAME_PROJECTCOLLECTION_UID,
							projectcollectionIdList));
			physicalEntities = ProjectMetadata.getDataServiceFacade()
					.getPhysicalDataContainer(query);
		} else {
			//if no noun exist, return an empty PhysicalDataContainer
			physicalEntities = ProjectMetadata.getDataServiceFacade()
					.getEmptyPhysicalDataContainer();
		}

		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.exiting(CLASSNAME, METHODNAME, physicalEntities);
		}

		return physicalEntities;

	}

	/**
	 * Resolve the noun part which has only the external identifier. The noun
	 * parts with external identifiers should be resolved to set the internal id
	 * in the object so that the read operation can be performed
	 * 
	 * @throws DataMediatorException 
	 * @see com.ibm.commerce.foundation.server.services.dataaccess.bom.mediator.AbstractChangeBusinessObjectMediatorImpl#resolveByExternalIdentifier(java.lang.Object)
	 */
	protected boolean resolveByExternalIdentifier(Object aNoun)
			throws DataMediatorException {

		final String METHODNAME = "resolveByExternalIdentifier";
		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.entering(CLASSNAME, METHODNAME, new Object[] { aNoun });
		}

		boolean resolveStatus = false;

		String extID = null;
		ProjectCollectionType projectcollectionNoun = (ProjectCollectionType) aNoun;

		String uniqueId = null;

		// get the uid and see whether it exists
		if (projectcollectionNoun.getProjectCollectionIdentifier() != null) {
			uniqueId = projectcollectionNoun.getProjectCollectionIdentifier()
					.getUniqueID();
		}

		if (uniqueId == null || uniqueId.length() == 0) {
			if (LoggingHelper.isTraceEnabled(LOGGER)) {
				LOGGER.logp(Level.FINE, CLASSNAME, METHODNAME,
						"need to resolve noun");
			}

			// get the external id
			if (projectcollectionNoun.getProjectCollectionIdentifier() != null
					&& projectcollectionNoun.getProjectCollectionIdentifier()
							.getExternalIdentifier() != null) {
				extID = projectcollectionNoun.getProjectCollectionIdentifier()
						.getExternalIdentifier().getName();
			}

			if (extID != null) {
				// construct selectionCriteria to find the ID from extID
				SelectionCriteria query = new SelectionCriteria();
				query
						.setXPathKey(ProjectFacadeConstants.XPATH_PROJECTCOLLECTION_NAME
								+ ProjectFacadeConstants.DOUBLE_CLOSE_XPATH);
				query
						.setAccessProfile(ProjectFacadeConstants.ACCESS_PROFILE_ID_RESOLVE);
				query
						.setXPathParameter(new RelationalExpression(
								ProjectFacadeConstants.NOUN_ATTRIBUTE_NAME_PROJECTCOLLECTION_NAME,
								extID));
				PhysicalDataContainer physicalEntites = ProjectMetadata
						.getDataServiceFacade().getPhysicalDataContainer(query);
				// if find the physical object of particular extID
				if (physicalEntites.getPhysicalObjects().size() > 0) {
					resolveStatus = true;
					Xprjcol projectcollectionEntity = (Xprjcol) physicalEntites
							.getPhysicalObjects().get(0);
					// set the id back to the Logical Business Object
					projectcollectionNoun.getProjectCollectionIdentifier()
							.setUniqueID(
									String.valueOf(projectcollectionEntity
											.getXprjcol_id()));

					if (LoggingHelper.isTraceEnabled(LOGGER)) {
						LOGGER.logp(LoggingHelper.DEFAULT_TRACE_LOG_LEVEL,
								CLASSNAME, METHODNAME,
								"The external identifier "
										+ projectcollectionEntity
												.getXprjcol_id()
										+ " into the noun");
					}
				}
				// if the external id is null
			} else {
				if (LoggingHelper.isTraceEnabled(LOGGER)) {
					LOGGER.logp(LoggingHelper.DEFAULT_TRACE_LOG_LEVEL,
							CLASSNAME, METHODNAME,
							"The external identifier is missing from the noun");
				}
			}

		}

		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.exiting(CLASSNAME, METHODNAME, new Boolean(resolveStatus));
		}

		return resolveStatus;

	}

	/**
	 * Checks if the object to be created is present in the physical data
	 * container. If the entity is already present then a validation error will
	 * be added to the return list
	 * 
	 * @param aNoun
	 *            The noun that is being requested to be changed
	 * @return List of errors; null if there are no errors
	 * @throws DataMediatorException
	 * 
	 * @see com.ibm.commerce.foundation.server.services.dataaccess.bom.mediator.ChangeBusinessObjectMediator#validateCreate(java.lang.Object)
	 */
	public List validateCreate(Object aNoun) throws DataMediatorException {

		final String METHODNAME = "validateCreate";
		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.entering(CLASSNAME, METHODNAME, new Object[] { aNoun });
		}

		List validationErrors = new ArrayList();

		ProjectCollectionType aProjectCollection = (ProjectCollectionType) aNoun;

		// check the name (external id)
		if (aProjectCollection.getProjectCollectionIdentifier() != null
				&& aProjectCollection.getProjectCollectionIdentifier()
						.getExternalIdentifier() != null
				&& aProjectCollection.getProjectCollectionIdentifier()
						.getExternalIdentifier().getName() != null) {

			String projectcollectionName = aProjectCollection
					.getProjectCollectionIdentifier().getExternalIdentifier()
					.getName();
			if (projectcollectionName == null
					|| projectcollectionName.trim().length() == 0) {
				// name cannot be empty
				ApplicationError validateError = new ApplicationError(
						ApplicationError.TYPE_GENERIC_ERROR,
						ProjectApplicationMessageKeys._APP_PROJECTCOLLECTION_NAME_EMPTY,
						null, LOGGER.getResourceBundleName());
				validationErrors.add(validateError);
			} else {

				if (LoggingHelper.isTraceEnabled(LOGGER)) {
					LOGGER.logp(Level.FINE, CLASSNAME, METHODNAME,
							"validate projectcollection name: "
									+ projectcollectionName);
				}

				// check for duplicate name existing in the database already
				if (findPhysicalEntity(aNoun) != null) {
					ApplicationError validateError = new ApplicationError(
							ApplicationError.TYPE_GENERIC_ERROR,
							ProjectApplicationMessageKeys._APP_PROJECTCOLLECTION_NAME_ALREADY_EXISTS,
							new Object[] { projectcollectionName }, LOGGER
									.getResourceBundleName());
					validationErrors.add(validateError);
				}

				// check for name length
				else if (ProjectMetadata.invalidStringLength(
						projectcollectionName,
						ProjectMetadata.PROJECTCOLLECTION_NAME_LENGTH_LIMIT)) {

					ApplicationError validateError = new ApplicationError(
							ApplicationError.TYPE_GENERIC_ERROR,
							ProjectApplicationMessageKeys._APP_PROJECTCOLLECTION_NAME_TOO_LONG,
							null, LOGGER.getResourceBundleName());
					validationErrors.add(validateError);
				}
			}
		} else {
			// check name cannot be empty
			ApplicationError validateError = new ApplicationError(
					ApplicationError.TYPE_GENERIC_ERROR,
					ProjectApplicationMessageKeys._APP_PROJECTCOLLECTION_NAME_EMPTY,
					null, LOGGER.getResourceBundleName());
			validationErrors.add(validateError);
		}

		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.exiting(CLASSNAME, METHODNAME, validationErrors);
		}
		return validationErrors;
	}

	/**
	 * Checks if the object to be changed is present in the physical data
	 * container. If the entity is not present then a validation error will be
	 * added to the return list
	 * 
	 * @param aNoun
	 *            The noun that is being requested to be changed
	 * @return List of errors; null if there are no errors
	 * @throws DataMediatorException
	 * 
	 * @see com.ibm.commerce.foundation.server.services.dataaccess.bom.mediator.ChangeBusinessObjectMediator#validateChange(java.lang.Object)
	 */
	public List validateChange(Object aNoun) throws DataMediatorException {

		final String METHODNAME = "validateChange";
		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.entering(CLASSNAME, METHODNAME, new Object[] { aNoun });
		}

		List validationErrors = new ArrayList();

		Xprjcol existingProjectCollection = (Xprjcol) findPhysicalEntity(aNoun);
		if (existingProjectCollection == null) {
			ApplicationError validateError = new ApplicationError(
					ApplicationError.TYPE_GENERIC_ERROR,
					ProjectApplicationMessageKeys._APP_PROJECTCOLLECTION_NOT_FOUND,
					new Object[] { aNoun }, LOGGER.getResourceBundleName());
			validationErrors.add(validateError);
		}

		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.exiting(CLASSNAME, METHODNAME, validationErrors);
		}

		return validationErrors;

	}

	/**
	 * Checks if the object to be removed is present in the physical data
	 * container. If the entity is not present then a validation error will be
	 * added to the return list
	 * 
	 * @param aNoun
	 *            The noun that is being requested to be deleted
	 * @return List of errors; null if there are no errors
	 * @throws DataMediatorException
	 * @see com.ibm.commerce.foundation.server.services.dataaccess.bom.mediator.ChangeBusinessObjectMediator#validateDelete(java.lang.Object)
	 */
	public List validateDelete(Object aNoun) throws DataMediatorException {

		final String METHODNAME = "validateDelete";
		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.entering(CLASSNAME, METHODNAME, new Object[] { aNoun });
		}

		List validationErrors = new ArrayList();
		if (findPhysicalEntity(aNoun) == null) {
			ApplicationError validateError = new ApplicationError(
					ApplicationError.TYPE_GENERIC_ERROR,
					ProjectApplicationMessageKeys._APP_PROJECTCOLLECTION_NOT_FOUND,
					new Object[] { aNoun }, LOGGER.getResourceBundleName());
			validationErrors.add(validateError);
		}

		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.exiting(CLASSNAME, METHODNAME, validationErrors);
		}
		return validationErrors;
	}

	/**
	 * This method updates the physical Xprjcol object with the user
	 * data set in the logical ProjectCollection noun. The name value pair
	 * mapping is set in the wc-businessobject-mediator XML file.
	 * 
	 * @param aLogicalEntity
	 *            The logical <code>ProjectCollectionType</code> SDO business
	 *            object.
	 * @param aPhysicalEntity
	 *            The physical <code>Xprjcol</code> SDO business
	 *            object.
	 * @throws AbstractApplicationException
	 */
	private void updateNameValuePairType(Object aLogicalEntity,
			Object aPhysicalEntity) throws AbstractApplicationException {
		final String METHODNAME = "updateNameValuePairType";
		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.entering(CLASSNAME, METHODNAME, new Object[] {
					aLogicalEntity, aPhysicalEntity });
		}

		if (((ProjectCollectionType) aLogicalEntity).getUserData() != null) {
			updateNameValuePairType(((ProjectCollectionType) aLogicalEntity)
					.getUserData().getUserDataField(), aLogicalEntity,
					aPhysicalEntity);
		}

		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.exiting(CLASSNAME, METHODNAME);
		}
	}
	
	/**
	 * This method will return the default instance of the Commerce Foundation
	 * Factory. This factory can be used to create objects defined by the
	 * Commerce Foundation.
	 * 
	 * @return The default Commerce Foundation Factory.
	 */
	protected CommerceFoundationFactory getCommerceFoundationFactory() {
		return CommerceFoundationFactory.eINSTANCE;
	}

}
