package com.ibm.commerce.rest.utils;

/*
 *-----------------------------------------------------------------
 * IBM Confidential
 *
 * OCO Source Materials
 *
 * WebSphere Commerce
 *
 * (C) Copyright IBM Corp. 2011, 2013
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has
 * been deposited with the U.S. Copyright Office.
 *-----------------------------------------------------------------
 */

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.ibm.commerce.foundation.client.facade.bod.AbstractBusinessObjectDocumentException;
import com.ibm.commerce.foundation.common.util.logging.LoggingHelper;
import com.ibm.commerce.foundation.rest.util.StoreConfigurationHelper;
import com.ibm.commerce.server.WcsApp;
import com.ibm.commerce.server.WebModuleConfig;

/**
 * Singleton class to build Resource URI for various resources.
 */
public class URLBuilder {
	
	/**
	 * IBM Copyright notice field.
	 */
	public static final String COPYRIGHT = com.ibm.commerce.copyright.IBMCopyright.SHORT_COPYRIGHT;

	

	private static final String CLASS_NAME = URLBuilder.class.getCanonicalName();
	
	private static final Logger LOGGER = LoggingHelper.getLogger(URLBuilder.class);
	
	private static final String DEFAULT_HOST_NAME 	= "localhost";
	private static final int DEFAULT_PORT 			= -1;
	private static final int DEFAULT_SECURE_PORT 	= -1;
	private static final String PORT_80 			= "80";
	private static final String PORT_443 			= "443";
	private static final String DEFAULT_CONTEXT_PATH = "/wcs/resources";
	private static final String DEFAULT_PREVIEW_CONTEXT_PATH = "/wcs/previewresources";
	private static final String STORE_ = "/store/";	
	private static final String DEFAULT_HTTP_PRE = "http://" + DEFAULT_HOST_NAME + DEFAULT_CONTEXT_PATH + STORE_;
	private static final String DEFAULT_HTTPS_PRE = "https://" + DEFAULT_HOST_NAME + DEFAULT_CONTEXT_PATH + STORE_;
	private static final String REST_PREVIEW_WEB_MODULE_NAME = "RestPreview";

	private static URLBuilder instance = null;

	private static URLBuilder previewInstance = null;
	
	private boolean dynamicBuilder = false;

	private String defaultHost = DEFAULT_HOST_NAME;

	private int defaultPort = DEFAULT_PORT;
	
	private int defaultSecurePort = DEFAULT_SECURE_PORT;

	private String defaultContextPath = DEFAULT_CONTEXT_PATH;
	
	private String restNonSSLPortConfigName = "restNonSSLPort";
	
	private String restSSLPortConfigName = "restSSLPort";
	

	/**
	 * Create an instance of the URLBuilder if it has not already been created.
	 * In a web module (server env), this is typically called from
	 * ServletContextListener with the appropriate values from the configuration
	 * In a test environment, this should be explicitly called once to set up
	 * the prefixes.
	 * 
	 * @param host
	 *            the externally visible host name
	 * @param port
	 *            the external port
	 * @param securePort 
	 * 			  the secure port
	 * @param contextPath
	 *            the context path to the web module
	 */
	public static void createInstance(String host, String port, String securePort,
			String contextPath) {
		final String METHODNAME = "createInstance(String host, String port, String securePort, String contextPath)";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { host, port, securePort, contextPath };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}
		
		if (null != instance) {
			LOGGER.logp(Level.WARNING, CLASS_NAME, "createInstance",
					"WARN_URL_BUILDER_INSTANCE_EXISTS", new Object[] {instance, instance.dynamicBuilder});
		} else {
			instance = new URLBuilder(host, port, securePort, contextPath);
		}
		
		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME);
		}
	}
	
	/**
	 * Creates the instance.
	 *
	 * @param contextPath the context path
	 */
	public static void createInstance(String contextPath) {
		final String METHODNAME = "createInstance(String contextPath)";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { contextPath };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}

		if (null != instance) {
			LOGGER.logp(Level.WARNING, CLASS_NAME, "createInstance",
					"WARN_URL_BUILDER_INSTANCE_EXISTS", new Object[] {instance, instance.dynamicBuilder});
		} else {
			instance = new URLBuilder(contextPath);			
		}
		
		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME);
		}
	}
	
	/**
	 * Creates the preview instance.
	 * 
	 * @param contextPath
	 *            the context path
	 */
	public static void createPreviewInstance(String contextPath) {
		final String METHODNAME = "createPreviewInstance(String contextPath)";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { contextPath };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}

		if (null != previewInstance) {
			LOGGER.logp(Level.WARNING, CLASS_NAME, "createPreviewInstance",
					"WARN_URL_BUILDER_INSTANCE_EXISTS", new Object[] {
					previewInstance, previewInstance.dynamicBuilder });
		} else {

			String inNonSSLPort = null;
			String inSSLPort = null;

			WebModuleConfig config = WcsApp.configProperties
					.getWebModule(REST_PREVIEW_WEB_MODULE_NAME);
			
			if (config != null) {
				inNonSSLPort = config.getNonSSLPort();
				inSSLPort = config.getSSLPort();
			}
			
			if (inNonSSLPort == null) {
				inNonSSLPort = PORT_80;
			}
			if (inSSLPort == null) {
				inSSLPort = PORT_443;
			}

			previewInstance = new URLBuilder(DEFAULT_HOST_NAME, inNonSSLPort,
					inSSLPort, contextPath);
			previewInstance.dynamicBuilder = true;
			previewInstance.restNonSSLPortConfigName = "previewRestNonSSLPort";
			previewInstance.restSSLPortConfigName = "previewRestSSLPort";
		}
		
		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME);
		}
	}

	/**
	 * Get the singleton instance of the URLBuilder. If createInstance has not
	 * been called prior to this call, then the singleton is initialized with
	 * the defaults.
	 *
	 * @return single instance of URLBuilder
	 */
	public static URLBuilder getInstance() {
		final String METHODNAME = "getInstance()";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}
		
		if (null == instance) {
			LOGGER.logp(Level.WARNING, CLASS_NAME, "getInstance",
					"WARN_URL_BUILDER_NOT_INITIALIZED");
			instance = new URLBuilder(DEFAULT_HOST_NAME, PORT_80, PORT_443,
					DEFAULT_CONTEXT_PATH);
		}
		
		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME,instance);
		}
		
		return instance;
	}
	
	/**
	 * Get the instance of the preview URLBuilder. If createPreviewInstance has not
	 * been called prior to this call, then the instance is initialized with
	 * the defaults.
	 *
	 * @return single instance of the preview URLBuilder
	 */
	public static URLBuilder getPreviewInstance() {
		final String METHODNAME = "getPreviewInstance()";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}
		
		if (null == previewInstance) {
			LOGGER.logp(Level.WARNING, CLASS_NAME, "getPreviewInstance",
					"WARN_URL_BUILDER_NOT_INITIALIZED");
			createPreviewInstance(DEFAULT_PREVIEW_CONTEXT_PATH);
		}
		
		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME,previewInstance);
		}
		
		return previewInstance;
	}
	
	/**
	 * Returns the URL builder depending on the isPreview flag.
	 * 
	 * @param isPreview
	 *            the preview URLBuilder is returned if set to true. Otherwise
	 *            the non-preview one is used.
	 * @return the URLBuilder requested.
	 */
	public static URLBuilder getInstance(boolean isPreview) {
		final String METHODNAME = "getInstance(boolean isPreview)";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] {isPreview };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}
		
		URLBuilder returnValue = null;

		if (isPreview) {
			returnValue = getPreviewInstance();
		} else {
			returnValue = getInstance();
		}
		
		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME,returnValue);
		}
		return returnValue;
	}

	private URLBuilder(String hostName, String port, String securePort, String contextPath) {
		final String METHODNAME = "URLBuilder(String hostName, String port, String securePort, String contextPath)";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { hostName, port, securePort, contextPath };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}
		
		this.defaultHost = hostName;
		if (!PORT_80.equals(port)) {
			this.defaultPort = Integer.parseInt(port);
		}
		if (!PORT_443.equals(securePort)) {
			this.defaultSecurePort = Integer.parseInt(securePort);
		}
		this.defaultContextPath = contextPath;
		
		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME);
		}
	}

	private URLBuilder(String contextPath) {
		final String METHODNAME = "URLBuilder(String contextPath)";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { contextPath };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}
		
		this.defaultContextPath = contextPath;
		this.dynamicBuilder = true;
		
		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME);
		}
	}
	

	/**
	 * Create the base URI.
	 *
	 * @param storeId the store id
	 * @return base uri - http://localhost/wcs/resources/store/10101
	 */
	public String createUri(String storeId) {
		final String METHODNAME = "createUri(String storeId)";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { storeId };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}
		
		String uri = DEFAULT_HTTP_PRE + storeId;
		try {
			uri = createUri(storeId, null, null);
		} catch (URISyntaxException e) {
			LOGGER.logp(Level.WARNING, CLASS_NAME, "createUri", "FAILED_TO_CREATE_URI", storeId);
		}
		
		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME, uri);
		}
		return uri;
	}

	/**
	 * Create a secure base URI.
	 *
	 * @param storeId the store id
	 * @return secure base URI - https://localhost/wcs/resources/store/10101
	 */
	public String createSecureUri(String storeId) {
		final String METHODNAME = "createSecureUri(String storeId)";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { storeId };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}
		
		String uri = DEFAULT_HTTPS_PRE + storeId;
		try {
			uri = createSecureUri(storeId, null, null);
		} catch (URISyntaxException e) {
			LOGGER.logp(Level.WARNING, CLASS_NAME, "createSecureUri", "FAILED_TO_CREATE_URI", storeId);
		}
		
		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME, uri);
		}
		return uri;
	}

	/**
	 * Create a HTTP URI from the storeId and the path relative to the store id.
	 * This method will encode the URI if the path segments contain reserved
	 * characters. For instance <code>createUri("10101", "/order/@self")</code>
	 * will return http://acme.com/wcs/resources/store/10101/order/@self
	 * <code>createUri("10101", "/category/Lounge Chair")</code> will return
	 * http://acme.com/wcs/resources/store/10101/category/Lounge%20Chair
	 *
	 * @param storeId the store identifier used in the URI
	 * @param hostName the host name. If it is null, the method will automatically get the web server host name.
	 * @param storeRelPath the store relative path to the resource (should start with a
	 * /)
	 * @return String the uri associated with the resource
	 * @throws URISyntaxException the uRI syntax exception
	 */
	public String createUri(String storeId, String hostName, String storeRelPath) throws URISyntaxException {
		final String METHODNAME = "createUri(String storeId, String hostName, String storeRelPath)";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { storeId, hostName, storeRelPath };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}
		
		StringBuilder path = new StringBuilder(defaultContextPath);
		path.append(STORE_).append(storeId);
		if (null != storeRelPath) {
			path.append(storeRelPath);
		}

		if (hostName == null) {
			hostName = getHost(storeId);
		}
		URI uri = new URI("http", null, hostName, getPort(storeId), path.toString(), null, null);
		String returnValue = uri.toString();
		
		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME, returnValue);
		}
		return returnValue;
	}

	/**
	 * Create a HTTPS URI from the storeId and the path relative to the store
	 * id. This method will encode the URI if the path segments contain reserved
	 * characters. For instance
	 * <code>createSecureUri("10101", "/order/@self")</code> will return
	 * https://acme.com/wcs/resources/store/10101/order/@self
	 * <code>createSecureUri("10101", "/category/Lounge Chair")</code> will
	 * return https://acme.com/wcs/resources/store/10101/category/Lounge%20Chair
	 *
	 * @param storeId the store identifier used in the URI
	 * @param hostName the host name. If it is null, the method will automatically get the web server host name.
	 * @param storeRelPath the store relative path to the resource (should start with a /, e.g., /cart/@self)
	 * @return String the uri associated with the resource
	 * @throws URISyntaxException the uRI syntax exception
	 */
	public String createSecureUri(String storeId, String hostName, String storeRelPath)
			throws URISyntaxException {
		final String METHODNAME = "createSecureUri(String storeId, String hostName, String storeRelPath)";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { storeId, hostName, storeRelPath };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}
		
		StringBuilder path = new StringBuilder(defaultContextPath);
		path.append(STORE_).append(storeId);
		if (null != storeRelPath) {
			path.append(storeRelPath);
		}

		if (hostName == null) {
			hostName = getHost(storeId);
		}
		URI uri = new URI("https", null, hostName, getSecurePort(storeId), path.toString(), null, null);
		String returnValue = uri.toString();
		
		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME, returnValue);
		}
		return returnValue;
	}

	/**
	 * Return the external host name associated with the store.
	 * 
	 * @param storeId
	 *            the store id
	 * @return the host
	 */
	public String getHost(String storeId) {
		final String METHODNAME = "getHost(String storeId)";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { storeId };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}

		String host = this.defaultHost;
		//add
		dynamicBuilder=false;
		if (dynamicBuilder) {
			try {
				host = StoreConfigurationHelper.getStoreConfiguration(storeId)
						.get("webServerHostName");
			} catch (AbstractBusinessObjectDocumentException abode) {
				Map<String, Object> errMap = GenericUtils
						.extractErrorFromBODException(null, abode);
				LOGGER.logp(Level.SEVERE, CLASS_NAME, METHODNAME,
						"ERR_EXCEPTION", new Object[] { METHODNAME,
								abode + ": " + errMap });
			}
		}

		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME, host);
		}
		return host;
	}
	
	/**
	 * Return the external HTTP port associated with the store if it is not 80, otherwise return -1.
	 *
	 * @param storeId the store id
	 * @return the port
	 */
	public int getPort(String storeId) {
		final String METHODNAME = "getPort(String storeId)";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { storeId };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}

		int port = this.defaultPort;
		if (dynamicBuilder) {
			try {
				String portStr = StoreConfigurationHelper.getStoreConfiguration(storeId).get(restNonSSLPortConfigName);
				if( portStr != null ){
					if (portStr.equals("80")) {
						port = -1; // This is the HTTP default port, no need to put in URI
					} else {
						port = Integer.parseInt(portStr);
					}
				}
			} catch (AbstractBusinessObjectDocumentException abode) {
	    		Map<String, Object> errMap = GenericUtils.extractErrorFromBODException(null, abode);
		    	LOGGER.logp(Level.SEVERE, CLASS_NAME, METHODNAME, "ERR_EXCEPTION", 
		    			new Object[]{METHODNAME, abode + ": " + errMap});	
			} catch (NumberFormatException nfe) {
				LoggingHelper.logUnexpectedException(LOGGER, CLASS_NAME, METHODNAME, nfe);
			}
		}	

		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME, port);
		}
		return port;
	}
	
	/**
	 * Return the secure (HTTPS) port associated with the store if it is not 443, if not return -1.
	 *
	 * @param storeId the store id
	 * @return the secure port
	 */
	public int getSecurePort(String storeId) {
		final String METHODNAME = "getSecureHost(String storeId)";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { storeId };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
		}

		int port = this.defaultSecurePort;
		if (dynamicBuilder) {
			try {
				String portStr = StoreConfigurationHelper.getStoreConfiguration(storeId).get(restSSLPortConfigName);
				if( portStr != null ){
					if (portStr.equals("443")) {
						port = -1;  // This is HTTPS default port, no need to put in URI
					} else {
						port = Integer.parseInt(portStr);
					}
				}
			} catch (AbstractBusinessObjectDocumentException abode) {
	    		Map<String, Object> errMap = GenericUtils.extractErrorFromBODException(null, abode);
		    	LOGGER.logp(Level.SEVERE, CLASS_NAME, METHODNAME, "ERR_EXCEPTION", 
		    			new Object[]{METHODNAME, abode + ": " + errMap});	
			} catch (NumberFormatException nfe) {
				LoggingHelper.logUnexpectedException(LOGGER, CLASS_NAME, METHODNAME, nfe);
			}
		}	
		
		if (entryExitTraceEnabled) {
			LOGGER.exiting(CLASS_NAME, METHODNAME, port);
		}
		return port;
	}
	
	/**
	 * Return context path.
	 *
	 * @return the context path
	 */
	public String getContextPath() {
		final String METHODNAME = "getContextPath()";
		boolean entryExitTraceEnabled = LoggingHelper
				.isEntryExitTraceEnabled(LOGGER);
		// boolean traceEnabled = LoggingHelper.isTraceEnabled(LOGGER);
		if (entryExitTraceEnabled) {
			Object[] parameters = new Object[] { };
			LOGGER.entering(CLASS_NAME, METHODNAME, parameters);
			LOGGER.exiting(CLASS_NAME, METHODNAME, defaultContextPath);
		}
		return defaultContextPath;
	}
}
