package com.mycompany.commerce.catalog.facade.server.services.search.expression.solr;

import java.util.Iterator;
import java.util.List;
import java.util.logging.Logger;

import com.ibm.commerce.context.globalization.GlobalizationContext;
import com.ibm.commerce.exception.ExceptionHandler;
import com.ibm.commerce.foundation.common.util.logging.LoggingHelper;
import com.ibm.commerce.foundation.internal.server.services.search.expression.SearchExpressionProvider;
import com.ibm.commerce.foundation.internal.server.services.search.expression.solr.AbstractSolrSearchExpressionProvider;
import com.ibm.commerce.foundation.server.services.businesscontext.ContextServiceFactory;
import com.ibm.commerce.foundation.server.services.dataaccess.SelectionCriteria;
import com.ibm.commerce.foundation.server.services.search.SearchServiceConstants;
import com.ibm.commerce.search.facade.client.SearchFacadeClient;

/**
 * This is a Solr specific implementation of the search expression provider for
 * expanding the search term into its synonyms using the search term association data
 * in WebSphere Commerce database instead of using the Solr <code>SynonymFactory</code>. 
 * The <code>_wcf.search.term</code> in the SelectionCriteria will be replaced
 * with the synonyms retrieved for the current language and store for other
 * downstream processing.
 */
public class SolrSearchTermAssociationExpressionProvider extends
		AbstractSolrSearchExpressionProvider implements
		SearchExpressionProvider {
	
	private static final String CLASSNAME = SolrSearchTermAssociationExpressionProvider.class
			.getName();
	
	private static final Logger LOGGER = LoggingHelper
			.getLogger(SolrSearchTermAssociationExpressionProvider.class);

	private String iComponentId = null;
	
	private final static String STR_SPACE = " ";

	/**
	 * Creates an instance of the search expression provider for a particular
	 * component. This instance can be used and re-used for future search
	 * requests.
	 * 
	 * @param componentId The target component.
	 */
	public SolrSearchTermAssociationExpressionProvider(String componentId) {
		super();
		iComponentId = componentId;
	}

	/**
	 * The business logic is performed here.
	 * 
	 * @param selectionCriteria The selection criteria.
	 * @throws RuntimeException if an exception occurs.
	 */
	public void invoke(SelectionCriteria selectionCriteria) throws RuntimeException {
		
		final String METHODNAME = "invoke(SelectionCriteria)";
		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.entering(CLASSNAME, METHODNAME,
					new Object[] { selectionCriteria });
		}
		
		super.invoke(selectionCriteria);

		// Query expression - _wcf.search.term
		String strSearchTerm = getControlParameterValue(SearchServiceConstants.CTRL_PARAM_SEARCH_TERM);
		if (LoggingHelper.isTraceEnabled(LOGGER)) {
			LOGGER.logp(LoggingHelper.DEFAULT_TRACE_LOG_LEVEL, CLASSNAME,
					METHODNAME, "Search term: " + strSearchTerm);
		}

		List<String> lstSynonyms = null;
		
		if (strSearchTerm != null && strSearchTerm.length() > 0) {
			
			SearchFacadeClient searchClient = new SearchFacadeClient();
			try {
				GlobalizationContext globCtxt = 
					(GlobalizationContext) ContextServiceFactory.getContextService()
						.findContext(GlobalizationContext.CONTEXT_NAME);
						
				String strLangId = globCtxt.getLanguageId().toString();
				
				lstSynonyms = searchClient.getSynonyms(strSearchTerm, strLangId);
			} catch (Exception e) {
				LOGGER.logp(java.util.logging.Level.WARNING, CLASSNAME, METHODNAME, 
					"Caught exception: " + ExceptionHandler.convertStackTraceToString(e)); 
			}
			
		}
		
		if ( lstSynonyms != null && !lstSynonyms.isEmpty() ) {
			Iterator<String> iterSearchTerm = lstSynonyms.iterator();
			
			StringBuffer sbSearchTerms = new StringBuffer(strSearchTerm);
			
			while (iterSearchTerm.hasNext()) {
				sbSearchTerms.append(STR_SPACE).append( iterSearchTerm.next() );
			}
			
			String strNewSearchTerms = sbSearchTerms.toString();
			
			if (LoggingHelper.isTraceEnabled(LOGGER)) {
				LOGGER.logp(LoggingHelper.DEFAULT_TRACE_LOG_LEVEL, CLASSNAME, METHODNAME, 
					"Substituting search term '" + strSearchTerm + "' with the search term '" + strNewSearchTerms + "'" );
			}
			setControlParameterValue(SearchServiceConstants.CTRL_PARAM_SEARCH_TERM, strNewSearchTerms);
		}
		
		if (LoggingHelper.isEntryExitTraceEnabled(LOGGER)) {
			LOGGER.exiting(CLASSNAME, METHODNAME);
		}
	}
}
