/**
 * This sample program is provided AS IS and may be used, executed, copied and modified without royalty payment by customer (a) for its own
 * instruction and study, (b) in order to develop applications designed to run with an IBM WebSphere product, either for customer's own internal use
 * or for redistribution by customer, as part of such an application, in customer's own products.
 * 
 * Products 5724-T52, 5724-T53, 5724-T57, 5724-T58, 5724-V03, 5724-T51, (C) COPYRIGHT International Business Machines Corp., 2010-2015
 * All Rights Reserved
 * Licensed Materials - Property of IBM
 */
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;

import com.ibm.json.java.JSONArray;
import com.ibm.json.java.JSONArtifact;
import com.ibm.json.java.JSONObject;

/**
 * Sample which illustrates a basic automation workflow in AppScan Enterprise: Log in, Loop through the applications, Loop through the issues of the applications, Submit defects to JIRA, Attach details of the issue to the submitted JIRA
 * defect, Attach a link back to the application from the submitted JIRA defect.
 * 
 * This file is intended to be customized before use. The static variables should be changed as appropriate and the disableCertificateValidation() call should be uncommented if you wish to temporarily disable certificate validation for
 * testing purposes.
 * 
 * Last Updated: April 30, 2015
 * 
 */
public class Sample {

	// AppScan Enterprise Configuration - Configure this section

	/**
	 * AppScan Enterprise REST Services Scheme
	 */
	private static String ASE_SCHEME = "https";

	/**
	 * AppScan Enterprise REST Services Host
	 */
	private static String ASE_HOST = "localhost";

	/**
	 * AppScan Enterprise REST Services Port
	 */
	private static int ASE_PORT = 9443;

	/**
	 * AppScan Enterprise REST Services Context Root
	 */
	private static String ASE_CONTEXT_ROOT = "/ase";

	/**
	 * User to log into ASE
	 */
	private static String ASE_USER = "username";

	/**
	 * Password to log into ASE
	 */
	private static String ASE_PASS = "password";

	/**
	 * Feature key to log into ASE.
	 */
	private static String ASE_FEATURE_KEY = "AppScanEnterpriseUser";

	/**
	 * AppScan's defect tracking system web application context root.
	 */
	private static String DTSWEB_CONTEXT_ROOT = "/dtsweb";

	// Defect tracking system configuration - Configure this section

	/**
	 * Defect tracking system (JIRA) URL.
	 */
	private static String JIRA_URL = "http://localhost:8080/";

	/**
	 * Project ID (JIRA key) of project in the defect tracking system that defects will be created in when this sample is executed.
	 */
	private static String JIRA_PROJECT_ID = "DEMO";

	/**
	 * Username to log into the defect tracking system (JIRA).
	 */
	private static String JIRA_USERNAME = "username";

	/**
	 * Password to log into the defect tracking system (JIRA).
	 */
	private static String JIRA_PASSWORD = "password";

	/**
	 * Defect tracking system's defect type to be created.
	 */
	private static String JIRA_DEFECT_TYPE = "3";

	/**
	 * Maximum number of issues or applications, which is used in the Range header. (An application intended for production use will also typically consider the result in the Content-Range header and use paging.)
	 */
	private static int MAX_NUMBER_OF_APPLICATIONS_OR_ISSUES = 10000;

	// Miscellaneous

	/**
	 * Boundary for making multipart requests.
	 */
	private static String MULTIPART_BOUNDARY = "sdfsdfsdfsdfsdfsdf";

	/**
	 * Persist cookies across requests.
	 */
	private static String cookieContainer = "";

	/**
	 * Value of the authorization header used by the defect tracking system (JIRA).
	 */
	private static String authorizationHeaderValue = null;

	/**
	 * The authorization header.
	 */
	private static String DTS_AUTHORIZATION_HEADER = "DTS-Authorization";

	/**
	 * Header used by ASE to prevent cross site scripting.
	 */
	private static String asc_xsrf_token;

	/**
	 * Map of issue attribute definitions.
	 */
	private static Map<String, String> issueAttributeDefinitions = null;

	/**
	 * The attribute id for the issuetype lookup.
	 */
	private static String issueTypeAttributeId = null;

	/**
	 * The request will return data in the specified format
	 */
	private static enum RETURN_FORMAT {
		JSON_ARRAY, JSON_OBJECT, ZIP
	}

	/**
	 * The specified method type will be used to perform the request
	 */
	private static enum REST_METHOD {
		GET, POST, PUT
	}

	/**
	 * The byte data returned by the request as well as the filename from the Content-Disposition header
	 */
	private static class Data {
		public byte[] byteArray;
		public String filename;
	}

	private static class RESTRequestResponse {
		public Object data;
		public String eTag;

		public RESTRequestResponse(Object data, String eTag) {
			this.data = data;
			this.eTag = eTag;
		}
	}

	/**
	 * Creates data suitable to be POSTed to the login request from the userId, password, and feature key.
	 * 
	 * @return JSONObject containing data suitable to be POSTed to the login request.
	 */
	private static JSONObject createLogin() {
		JSONObject jsonObject = new JSONObject();
		jsonObject.put("userId", ASE_USER);
		jsonObject.put("password", ASE_PASS);
		jsonObject.put("featureKey", ASE_FEATURE_KEY);
		return jsonObject;
	}

	/**
	 * Constructs an Authorization header suitable for BASIC HTTP authentication.
	 * 
	 * @return String Authorization header suitable for BASIC HTTP authentication.
	 */
	private static String getAuthorizationHeader() {
		if (authorizationHeaderValue == null) {
			byte[] encodedToBase64 = Base64.encodeBase64((JIRA_USERNAME + ":" + JIRA_PASSWORD).getBytes());
			authorizationHeaderValue = "Basic " + new String(encodedToBase64);
		}
		return authorizationHeaderValue;
	}

	/**
	 * Obtains the issue attribute definitions and caches it in a map.
	 * 
	 * @return Map<String, String> of issue attribute definitions, where the key is the ID and the value is the display name.
	 * @throws Exception
	 */
	private static void obtainIssueAttributeDefinitions() throws Exception {
		issueAttributeDefinitions = new HashMap<String, String>();

		JSONObject responseObject = (JSONObject) sendRESTRequest(new URI(ASE_SCHEME, null, ASE_HOST, ASE_PORT, ASE_CONTEXT_ROOT + "/api/issueattributedefinitions", null, null).toURL(), null, REST_METHOD.GET, RETURN_FORMAT.JSON_OBJECT,
				null, null).data;

		JSONArray responseArray = (JSONArray) responseObject.get("attributeDefColl");
		for (int i = 0; i < responseArray.size(); i++) {
			JSONObject jsonObject = (JSONObject) responseArray.get(i);
			Long definitionId = (Long) jsonObject.get("id");
			String definitionName = (String) jsonObject.get("name");

			String lookup = (String) jsonObject.get("lookup");
			if ("issuetype".equals(lookup)) {
				issueTypeAttributeId = definitionId.toString();
			}

			issueAttributeDefinitions.put(definitionId.toString(), definitionName);
		}

	}

	/**
	 * Creates defects for an ASE application. One defect is created per issue in the ASE application.
	 * 
	 * @param appName
	 *            String name of the ASE application
	 * @param appId
	 *            String ID of the ASE application
	 * 
	 * @throws Exception
	 */
	private static void createDefectsForApplication(String appName, String appId) throws Exception {
		print("Creating issues for the application named " + appName + " with id " + appId);

		JSONObject jsonObject = new JSONObject();
		appName = appName.replace("=", "\\=");
		appName = appName.replace(",", "\\,");
		appName = appName.replace("\\", "\\\\");

		jsonObject.put("query", "Application Name=" + appName);

		JSONArray responseArray = (JSONArray) sendRESTRequest(new URI(ASE_SCHEME, null, ASE_HOST, ASE_PORT, ASE_CONTEXT_ROOT + "/api/issues", "query=Application Name=" + appName, null).toURL(), null, REST_METHOD.GET,
				RETURN_FORMAT.JSON_ARRAY, null, getRangeHeader()).data;

		for (int i = 0; i < responseArray.size(); i++) {
			JSONObject aDefect = (JSONObject) responseArray.get(i);
			StringBuffer defectDescription = new StringBuffer();
			@SuppressWarnings("unchecked")
			Iterator<Entry<String, String>> defectPropertiesIterator = aDefect.entrySet().iterator();
			String issueType = null, issueId = null;
			while (defectPropertiesIterator.hasNext()) {
				Entry<String, String> property = defectPropertiesIterator.next();
				String key = issueAttributeDefinitions.get(property.getKey());
				String value = property.getValue();

				if (property.getKey().equals(issueTypeAttributeId)) {
					issueType = value;
				} else if (property.getKey().equals("id")) {
					issueId = value;
				}

				if (key == null) {
					key = property.getKey();
				}
				if (value == null) {
					value = "Unknown value";
				}
				if (defectDescription.length() > 0) {
					defectDescription.append("\n");
				}
				defectDescription.append(key + ": " + value);

			}

			System.out.println(defectDescription);

			JSONObject newDefect = new JSONObject();
			newDefect.put("defectTrackingSystem", "JIRA");
			newDefect.put("url", JIRA_URL);
			newDefect.put("projectId", JIRA_PROJECT_ID);
			newDefect.put("title", appName + (issueType != null ? " - " + issueType : ""));
			newDefect.put("description", defectDescription.toString());
			newDefect.put("defectTypeId", JIRA_DEFECT_TYPE);

			JSONObject responseObject = (JSONObject) sendRESTRequest(new URI(ASE_SCHEME, null, ASE_HOST, ASE_PORT, DTSWEB_CONTEXT_ROOT + "/api/defects", null, null).toURL(), newDefect, REST_METHOD.POST, RETURN_FORMAT.JSON_OBJECT, null,
					null).data;

			String newDefectId = (String) responseObject.get("id");

			if (newDefectId != null) {
				attach(newDefectId, appId, issueId);
				link(newDefectId, appId, appName, issueId);
				updateExternalId("JIRA:" + newDefectId, appId, issueId);
			}

		}

		System.out.println(responseArray.serialize());
	}

	/**
	 * Attaches an attachment to an existing JIRA defect with the id jiraId. In this sample, a zip containing exported issue data is added as an attachment.
	 * 
	 * @param jiraId
	 *            String JIRA defect ID
	 * @param appId
	 *            String ASE application ID
	 * @param issueId
	 *            String ASE issue ID
	 * @throws Exception
	 */
	private static void attach(String jiraId, String appId, String issueId) throws Exception {
		MultipartEntityBuilder meb = MultipartEntityBuilder.create().setBoundary(MULTIPART_BOUNDARY);
		meb.setMode(HttpMultipartMode.BROWSER_COMPATIBLE);

		JSONObject existingDefect = new JSONObject();
		existingDefect.put("defectTrackingSystem", "JIRA");

		existingDefect.put("url", JIRA_URL);
		existingDefect.put("projectId", JIRA_PROJECT_ID);
		existingDefect.put("id", jiraId);

		meb.addTextBody("jsonBody", existingDefect.serialize());

		Data zip = (Data) sendRESTRequest(new URI(ASE_SCHEME, null, ASE_HOST, ASE_PORT, ASE_CONTEXT_ROOT + "/api/issues/details", "appId=" + appId + "&ids=" + issueId, null).toURL(), null, REST_METHOD.GET, RETURN_FORMAT.ZIP, null, null).data;

		meb.addBinaryBody("jsonBody", zip.byteArray, ContentType.create("application/zip"), (zip.filename != null ? zip.filename : issueId + ".zip"));

		sendRESTRequest(new URI(ASE_SCHEME, null, ASE_HOST, ASE_PORT, DTSWEB_CONTEXT_ROOT + "/api/defects/attachments", null, null).toURL(), meb.build(), REST_METHOD.POST, null, null, null);

	}

	/**
	 * Adds a link to an existing JIRA defect with the id jiraId. In this sample, a link back to the ASE application in the ASE instance is added, and it is filtered by the relevant issue ID.
	 * 
	 * @param jiraId
	 *            String JIRA defect ID
	 * @param appId
	 *            String ASE application ID
	 * @param appName
	 *            String ASE application name
	 * @param issueId
	 *            String ASE issue ID
	 * @throws Exception
	 */
	private static void link(String jiraId, String appId, String appName, String issueId) throws Exception {
		JSONObject existingDefect = new JSONObject();
		existingDefect.put("defectTrackingSystem", "JIRA");

		existingDefect.put("url", JIRA_URL);
		existingDefect.put("projectId", JIRA_PROJECT_ID);
		existingDefect.put("id", jiraId);

		JSONArray links = new JSONArray();
		JSONObject aLink = new JSONObject();
		aLink.put("name", appName + " issue " + issueId);
		aLink.put("url", new URI(ASE_SCHEME, null, ASE_HOST, ASE_PORT, "/ase/api/pages/applications.html", null, "appProfile/" + appId + "/issues/*0=" + issueId).toString());
		links.add(aLink);
		existingDefect.put("links", links);

		sendRESTRequest(new URI(ASE_SCHEME, null, ASE_HOST, ASE_PORT, DTSWEB_CONTEXT_ROOT + "/api/defects/links", null, null).toURL(), existingDefect, REST_METHOD.POST, null, null, null);

	}

	private static void updateExternalId(String jiraId, String appId, String issueId) throws Exception {
		RESTRequestResponse response = sendRESTRequest(new URI(ASE_SCHEME, null, ASE_HOST, ASE_PORT, ASE_CONTEXT_ROOT + "/api/issues/" + issueId + "/application/" + appId, null, null).toURL(), null, REST_METHOD.GET,
				RETURN_FORMAT.JSON_OBJECT, null, null);
		JSONObject responseObject = (JSONObject) response.data;

		JSONObject attributeCollectionObject = (JSONObject) responseObject.get("attributeCollection");
		JSONArray attributeArray = (JSONArray) attributeCollectionObject.get("attributeArray");
		for (int i = 0; i < attributeArray.size(); i++) {
			JSONObject attributeObject = (JSONObject) attributeArray.get(i);
			if ("externalid".equals(attributeObject.get("lookup"))) {
				JSONArray valueArray = (JSONArray) attributeObject.get("value");
				valueArray.add(jiraId);

				JSONObject postObject = new JSONObject();
				postObject.put("appReleaseId", appId);
				postObject.put("issueId", issueId);
				JSONArray attributes = new JSONArray();
				JSONObject attributeArrayOuterObject = new JSONObject();
				attributeArrayOuterObject.put("attributeArray", attributes);
				JSONObject attributeArrayInnerObject = new JSONObject();
				attributeArrayInnerObject.put("name", attributeObject.get("name"));
				attributeArrayInnerObject.put("attributeType", attributeObject.get("attributeType")); // 3 is eATTMultiValueTextBox
				attributeArrayInnerObject.put("issueAttributeDefinitionId", attributeObject.get("issueAttributeDefinitionId"));
				attributeArrayInnerObject.put("value", valueArray);
				attributes.add(attributeArrayInnerObject);
				postObject.put("attributeCollection", attributeArrayOuterObject);

				// if this request fails, and this was production code, you could retry a few times.
				// it could fail if there were simultaneous updates
				sendRESTRequest(new URI(ASE_SCHEME, null, ASE_HOST, ASE_PORT, ASE_CONTEXT_ROOT + "/api/issues/" + issueId, null, null).toURL(), postObject, REST_METHOD.PUT, RETURN_FORMAT.JSON_OBJECT, response.eTag, null);

				return;
			}
		}
	}

	/**
	 * The main method logs in and loops through each application to create a JIRA defect for each issue in each application.
	 * 
	 * @param args
	 *            String[] Ignored
	 */
	public static void main(String[] args) {

		// Server needs TLSv1 or higher
		System.setProperty("https.protocols", "TLSv1");

		// FOR TESTING PURPOSE ONLY - You can disable certificate validation
		// disableCertificateValidation();

		try {

			// Authenticate
			print("Login");

			// Log in
			JSONObject responseObject = (JSONObject) sendRESTRequest(new URI(ASE_SCHEME, null, ASE_HOST, ASE_PORT, ASE_CONTEXT_ROOT + "/api/login", null, null).toURL(), createLogin(), REST_METHOD.POST, RETURN_FORMAT.JSON_OBJECT, null, null).data;
			if (responseObject == null) {
				print("An error occurred while logging in");
				return;
			}

			asc_xsrf_token = (String) responseObject.get("sessionId");
			print("The asc_xsrf_token header, which must included in all subsequent headers for requests to ASE in order to prevent cross-site scripting, is " + asc_xsrf_token);

			JSONArray responseArray = (JSONArray) sendRESTRequest(new URI(ASE_SCHEME, null, ASE_HOST, ASE_PORT, ASE_CONTEXT_ROOT + "/api/applications", null, null).toURL(), null, REST_METHOD.GET, RETURN_FORMAT.JSON_ARRAY, null,
					getRangeHeader()).data;
			if (responseArray == null) {
				print("An error occurred while making the request to obtain the applications");
				return;
			}
			print("The list of applications was obtained: " + responseArray.serialize());

			obtainIssueAttributeDefinitions();

			print("For each issue in each application, we will create a defect");

			for (int i = 0; i < responseArray.size(); i++) {
				JSONObject jsonObj = (JSONObject) responseArray.get(i);
				createDefectsForApplication((String) jsonObj.get("name"), (String) jsonObj.get("id"));
			}

		} catch (Exception e) {
			print("An error occurred.");
			e.printStackTrace();
		}
	}

	private static void disableCertificateValidation() {
		TrustManager[] trustAllCerts = new TrustManager[] { new X509TrustManager() {
			public java.security.cert.X509Certificate[] getAcceptedIssuers() {
				return null;
			}

			public void checkClientTrusted(X509Certificate[] certs, String authType) {
			}

			public void checkServerTrusted(X509Certificate[] certs, String authType) {
			}

		} };

		SSLContext sc = null;
		try {
			sc = SSLContext.getInstance("SSL");
		} catch (NoSuchAlgorithmException e1) {
		}

		try {
			sc.init(null, trustAllCerts, new java.security.SecureRandom());
		} catch (KeyManagementException e1) {
		}

		HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());

		// Create all-trusting host name verifier
		HostnameVerifier allHostsValid = new HostnameVerifier() {
			public boolean verify(String hostname, SSLSession session) {
				return true;
			}
		};
		// Install the all-trusting host verifier
		HttpsURLConnection.setDefaultHostnameVerifier(allHostsValid);
	}

	/**
	 * Prints to the console
	 * 
	 * @param message
	 */
	private static void print(String message) {
		System.out.println(message);
	}

	/**
	 * Sends a REST request
	 * 
	 * @param url
	 *            URL the request is made on
	 * @param data
	 *            HttpEntity or JSONArtifact
	 * @param method
	 *            REST_METHOD to specify POST or GET
	 * @param ret
	 *            RETURN_FORMAT to specify how the returned data should be interpreted (JSON or ZIP)
	 * @return The result of the request, interpreted in the format specified by ret.
	 * 
	 * @throws Exception
	 */
	private static RESTRequestResponse sendRESTRequest(URL url, Object data, REST_METHOD method, RETURN_FORMAT ret, String ifMatchHeader, String rangeHeader) throws Exception {

		HttpURLConnection conn = (HttpURLConnection) url.openConnection();
		conn.setDoInput(true);

		if (ret == RETURN_FORMAT.JSON_ARRAY || ret == RETURN_FORMAT.JSON_OBJECT) {
			conn.addRequestProperty(HttpHeaders.ACCEPT, ContentType.APPLICATION_JSON.toString());
		} else if (ret == RETURN_FORMAT.ZIP) {
			conn.addRequestProperty(HttpHeaders.ACCEPT, "application/zip");
		}

		if (data instanceof HttpEntity) {
			conn.addRequestProperty(HttpHeaders.CONTENT_TYPE, "multipart/form-data; boundary=" + MULTIPART_BOUNDARY);
		} else {
			conn.addRequestProperty("Content-Type", ContentType.APPLICATION_JSON.toString());
		}

		if (asc_xsrf_token != null) {
			conn.setRequestProperty("asc_xsrf_token", asc_xsrf_token);
		}

		if (cookieContainer.length() > 0) {
			conn.setRequestProperty("Cookie", cookieContainer);
		}

		if (ifMatchHeader != null) {
			conn.setRequestProperty(HttpHeaders.IF_MATCH, ifMatchHeader);
		}

		if (rangeHeader != null) {
			conn.setRequestProperty(HttpHeaders.RANGE, rangeHeader);
		}

		// there's no point to send this header for ASE requests
		if (url.getPath().startsWith(DTSWEB_CONTEXT_ROOT + "/")) {
			conn.setRequestProperty(DTS_AUTHORIZATION_HEADER, getAuthorizationHeader());
		}

		conn.setRequestMethod(method.toString());
		print(conn.getRequestMethod() + " " + url.toString());

		if (data != null) {
			String json = null;
			if (data instanceof JSONArtifact) {
				json = ((JSONArtifact) data).serialize();
				print(json);
			}
			conn.setDoOutput(true);
			OutputStreamWriter wr = null;
			try {
				if (data instanceof HttpEntity) {
					((HttpEntity) data).writeTo(conn.getOutputStream());
				} else if (data instanceof JSONArtifact) {
					wr = new OutputStreamWriter(conn.getOutputStream());
					wr.write(json);
					wr.flush();
				} else {
					throw new IllegalArgumentException("The data parameter must be a JSONArtifact or a HttpEntity, but it was a " + data.getClass().getName());
				}
			} finally {
				if (wr != null) {
					wr.close();
				}
			}
		}
		try {
			int statusCode = conn.getResponseCode();
			if (statusCode < 200 || statusCode >= 300) {
				System.out.println("The request returned the error code " + statusCode);
				InputStream errorStream = conn.getErrorStream();
				if (errorStream != null) {
					System.out.println(IOUtils.toString(conn.getErrorStream()));
				}
				return null;
			}
			if (ret == null) {
				return null;
			} else if (ret == RETURN_FORMAT.JSON_ARRAY) {
				return new RESTRequestResponse(JSONArray.parse(conn.getInputStream()), conn.getHeaderField(HttpHeaders.ETAG));
			} else if (ret == RETURN_FORMAT.JSON_OBJECT) {
				return new RESTRequestResponse(JSONObject.parse(conn.getInputStream()), conn.getHeaderField(HttpHeaders.ETAG));
			}
			Data zip = new Data();
			zip.byteArray = IOUtils.toByteArray(conn.getInputStream());
			String contentDisposition = conn.getHeaderField("Content-Disposition");

			int idx1 = contentDisposition.indexOf("filename=\"");
			if (idx1 != -1) {
				int startIdx = idx1 + 10 /* length of filename=" */;
				if (contentDisposition.length() > startIdx) {
					int idx2 = contentDisposition.indexOf("\"", startIdx);
					if (idx2 != -1) {
						zip.filename = contentDisposition.substring(startIdx, idx2);
						System.out.println("The filename returned in the Content-Disposition header was " + zip.filename);
					}
				}

			}

			return new RESTRequestResponse(zip, conn.getHeaderField(HttpHeaders.ETAG));

		} finally {
			// Update cookies
			Map<String, List<String>> responseHeaders = conn.getHeaderFields();
			List<String> cookies = responseHeaders.get("Set-Cookie");
			if (cookies != null && cookies.size() > 0) {
				for (String cookie : cookies) {
					if (cookieContainer.length() > 0)
						cookieContainer += "; ";

					int indexOfSemicolon = cookie.indexOf(';');
					if (indexOfSemicolon > -1) {
						cookie = cookie.substring(0, indexOfSemicolon);
					}

					cookieContainer += cookie;
				}
			}
			conn.disconnect();
		}
	}

	private static String getRangeHeader() {
		return "items=0-" + MAX_NUMBER_OF_APPLICATIONS_OR_ISSUES;
	}

}
